package BBS::Timer;
our $NOLOAD = 1;
#===========================================================
# BBS.pm サブモジュール 【 タイマ管理 】
#
our $VERSION = [
	'0001.20250322.2045',		# 正式公開版
];
#===========================================================
# [説明]
#     本モジュールは汎用タイマを提供します。
#===========================================================
use utf8;
use strict;
use warnings;
use Time::HiRes qw( gettimeofday tv_interval );
use Data::Dumper;

#==================================================
# ●コンストラクタ
# 
# [書式]
#     obj = new( interval )
#
# [引数]
#     interval = インターバル値( 1 = 0.000001 秒 )(任意)
#
# [返り値]
#     obj = オブジェクト
# 
# [説明]
#     interval が設定されていると、現在時刻からのタイムアウト値を設定した状態でオブジェクトを作成します。
#==================================================
sub new {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::new';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $class = shift;
	my $interval = shift;			# タイムアウト値
	my $self = {};
	bless($self, $class);
	$self->{'Timeout'} = $interval;
	$self->_init();
	return $self;
}

#==================================================
# ●属性の初期化
# 
# [書式]
#     _init()
# 
# [引数]
#     なし
#
# [返り値]
#     なし
#
# [説明]
#     この関数は new() より呼び出すもので、直接呼び出し禁止です。
#     オブジェクトの属性を初期化します。
#==================================================
sub _init {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::_init';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;
	my $interval = $self->{'Timeout'};		# タイムアウト値
	my $now = [ gettimeofday ];				# 現在時刻

	# ◆インターバル値が定義されている
	if ( defined($interval) ) {
		if ( $interval =~ /^\d+$/ ) {											# インターバル値が数値
			# printf "\n** interval=(%d)", $interval;									# ????
			my $timeout = addtime( $now, $interval );								# タイムアウト値を取得
			$self->{'Timeout'} = $timeout if ( defined($timeout) );					# 取得したタイムアウト値
			# my ( $sec, $msec ) = @{ $timeout };										# ????
			# my $t = sprintf "%d.%06d", $sec, $msec;									# ????
			# printf "\n** timeout=(%.06f)", $t;										# ????
		}
	}
	# ◆インターバル値が未定義
	else {
		delete( $self->{'Timeout'} );
	}
	$self->{'Start'} = $now;				# 起点時刻を保存
}

#==================================================
# ●エラー値をセットまたはエラー値を返す
# 
# [書式]
#     err = err();					【 エラー値参照 】
#     err( err );					【 エラー値セット 】
# 
# [引数]
#     err : エラー値					【 セット時 】
# 
# [返り値]
#     err = エラー値					【 参照時 】
# 
# [説明]
#     ルーチン内のエラーをセットまたは参照します。
#==================================================
sub err {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::err';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;
	return $self->{'err'} if ( $#_ < 0 );			# 引数がなければエラーを返す
	$self->{'err'} = shift;							# 引数があればエラーをセット
}

#==================================================
# ●起点時間のリセット
# 
# [書式]
#     r = reset( interval )
# 
# [引数]
#     interval = インターバル値(任意)( 1 = 0.000001 秒 )
#
# [返り値]
#     r =
#         (失敗) : 0
#         (成功) : 1
# 
# [説明]
#     起点時間をリセットします。
#     インターバル値が指定されていると、セットしなおした起点時間からインターバル値後のタイムアウト値がセットされます。
#==================================================
sub reset {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::reset';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;
	my $interval = shift;					# タイムアウト値
	$self->{'Timeout'} = $interval;
	return $self->_init();
}

#==================================================
# ●現在時刻を返す
# 
# [書式]
#     r = now()
# 
# [引数]
#     なし
#
# [返り値]
#     r = 現在時刻(gettimeofday形式)
# 
# [説明]
#     
#==================================================
sub now {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::now';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;
	return [ gettimeofday ];						# 現在時刻
}

#==================================================
# ●起点時刻からの経過時間を返す
# 
# [書式]
#     r = elapse()
# 
# [引数]
#     なし
#
# [返り値]
#     r = 経過時間(gettimeofday形式)
# 
# [説明]
# 
# [メモ]
#     数値にする場合は返り値に sprintf("%d.%06d") を通す（安全でないかも）。
#==================================================
sub elapse {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::elapse';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;
	my $now = [ gettimeofday ];						# 現在時刻(起点時刻)
	my $start = $self->{'Start'};					# 

	# printf "\n** start = [ %d, %d ] ", $start->[0], $start->[1];		# ????
	# printf "\n**   now = [ %d, %d ] ", $now->[0],   $now->[1];		# ????

	my( $sec, $msec ) = @{ timediff( $start, $now ) };
	# printf "\n** sec=(%d), msec=(%d)", $sec, $msec;			# ????
	# $msec = sprintf( "%06s", $msec );
	# my $r = sprintf( "%s.%s", $sec, $msec );
	# return $r;
	return [ $sec, $msec ];
}

#==================================================
# ●タイムアウトまでの残り時間を返す
# 
# [書式]
#     r = timeleft()
# 
# [引数]
#     なし
#
# [返り値]
#     r = 
#         (失敗) : undef (timeout値が未定義)
#         (成功) : 残り時間 (gettimeofday形式)
#                 (タイムアウト時: [ 0, 0 ])
# 
# [説明]
# 
# [メモ]
#     数値にする場合は返り値に sprintf("%d.%06d") を通す（安全でないかも）。
#==================================================
sub timeleft {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::timeleft';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;
	my $now = [ gettimeofday ];				# 起点時刻
	my $timeout = $self->{'Timeout'};		# 終点時刻
	return unless ( defined($timeout) );									# タイムアウト値が未定義(undef)

	# printf "\n**     now = [ %d, %d ] ", $now->[0],   $now->[1];			# ????
	# printf "\n** timeout = [ %d, %d ] ", $timeout->[0], $timeout->[1];	# ????

	my( $sec, $msec ) = @{ timediff( $now, $timeout ) };
	# printf "\n** sec=(%d), msec=(%d)", $sec, $msec;			# ????
	# $msec = sprintf( "%06s", $msec );
	# my $r = sprintf( "%s.%s", $sec, $msec );
	# return $r;

	return [ $sec, $msec ] if ( $self->timeout() == 0 );					# タイムアウトしていなければ残り時間を返す
	return [ 0, 0 ];														# タイムアウトしていたら０を返す
}

#==================================================
# ●タイムアウト状態を返す
# 
# [書式]
#     timeout()
# 
# [引数]
#     なし
# 
# [返り値]
#     r =
#         -1 : タイムアウト値が定義されていない
#          0 : タイムアウトしていない
#          1 : タイムアウトしている
# 
# [説明]
# 
#==================================================
sub timeout {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::timeout';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;
	return -1 unless ( exists( $self->{'Timeout'} ) );						# タイムアウト値が未定義(-1)

	my $i = $self->{'Timeout'};				# 終点時刻（タイムアウト値）
	return -1 unless ( defined($i) );		# 終点時刻が未定義(-1)
	my $j = [ gettimeofday ];				# 起点時刻（現在時刻）
	my $sec  = 0;							# 秒
	my $usec = 1;							# マイクロ秒
	my $r = 0;

	# printf "\n** i=[ %d, %d ] ", $i->[$sec], $i->[$usec];				# ????
	# printf "\n** j=[ %d, %d ] ", $j->[$sec], $j->[$usec];				# ????

	# 現在秒が設定値を超えている(1)
	$r = 1 if ( $j->[$sec] > $i->[$sec] );

	# 現在秒が設定値と同じかつ、現在マイクロ秒が設定値を超えている(1)
	$r = 1 if ( ( $j->[$sec] == $i->[$sec] ) && ( $j->[$usec] > $i->[$usec] ) );

	return $r;
}

#==================================================
# ●起点時刻に時差を加算した時刻を返す
# 
# [書式]
#     r = addtime( i, x )
#
# [引数]
#     i : 起点時刻 (gettimeofdayの形式)
#     x : 時差 (1 = 0.000001秒)
#
# [返り値]
#     r = 終点時刻（gettimeofdayの形式）
# 
# [説明]
# 
#==================================================
sub addtime {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::addtime';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	# printf "\n\n==( %s::addtime [ %s ] )\n", __PACKAGE__, Debug::_caller(caller) if ( Debug::envchk( '<Timer:addtime' ) );						# ????
	my $i = shift;				# 起点時刻（gettimeofday形式）
	my $x = shift;				# 時差（マイクロ秒）
	my $sec  = 0;				# 秒
	my $usec = 1;				# マイクロ秒

	return unless ( defined($i) );										# 起点時刻が未指定(undef)
	return if ( $i->[$sec] !~ /^\d+$/ );								# 起点時刻（秒）が数値でない(undef)
	return if ( $i->[$usec] !~ /^\d+$/ );								# 起点時刻（マイクロ秒）が数値でない(undef)
	return if ( ( defined($x) == 0 ) || ( $x !~ /^\d+$/ ) );			# 時差が未指定または指定値が数値でない(undef)

	my $j = [ ];				# 時差（gettimeofday形式）
	my $k;						# 終点時刻（gettimeofday形式）

	# 時差をgettimeofday形式に変換
	$j = ( $x >= 1000000 )
			? [ int( $x / 1000000 ), ( $x % 1000000 ) ]					# １秒以上
			: [ 0, $x ];												# １秒未満

	# printf "\n** ts = [ %d, %d ] ", $i->[$sec], $i->[$usec];			# ???? 起点時刻
	# printf "\n** td = [ %d, %d ] ", $j->[$sec], $j->[$usec];			# ???? 時差

	# 時間を合算
	$k->[$sec] = ( $i->[$sec] + $j->[$sec] );							# 秒の合算
	my $usec_total = $i->[$usec] + $j->[$usec];							# マイクロ秒の合算
	# ◆マイクロ秒の合算値が１秒を超える
	if ( ( $usec_total ) >= 1000000 ) {
		$k->[$sec]++;														# 秒＋１
		$k->[$usec] = ( $usec_total % 1000000 );							# マイクロ秒の合算値を１秒で割った余り
	}
	# ◆マイクロ秒の合算値が１秒未満
	else {
		$k->[$usec] = ( $usec_total );										# マイクロ秒の合算値
	}
	# printf "\n** te = [ %d, %d ]\n", $k->[$sec], $k->[$usec];			# ???? 終点時刻
	return $k;
}

#==================================================
# ●２つの時刻の時間差を返す
# 
# [書式]
#     k = timediff( i, j )
#
# [引数]
#     i : 起点時刻（gettimeofday形式）
#     j : 終点時刻（gettimeofday形式）
#
# [返り値]
#     k =
#         (失敗) : undef (引数が未定義)
#         (失敗) : [ -1, 0 ] (起点時刻が終点時刻を超えている)
#         (成功) : 時間差（gettimeofdayの形式）
# 
# [説明]
#
#==================================================
sub timediff {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::timediff';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $i_ = shift;							# 起点時刻（gettimeofday形式）
	my $j_ = shift;							# 終点時刻（gettimeofday形式）
	return unless ( defined($i_) );			# 起点時刻が未定義(undef)
	return unless ( defined($j_) );			# 終点時刻が未定義(undef)

	my @i = @{ $i_ };
	my @j = @{ $j_ };
	my @k = ();								# 時間差（gettimeofday形式）
	my $usec = 1;							# マイクロ秒
	my $sec  = 0;							# 秒

	# printf "\n** i = [ %d, %d ] ", $i[$sec], $i[$usec];				# ????
	# printf "\n** j = [ %d, %d ] ", $j[$sec], $j[$usec];				# ????
	# print "\n";														# ????
	return [ -1, 0 ] if ( $j[$sec] < $i[$sec] );

	# (1) １秒以上の差を算出
	$k[$sec] = $j[$sec] - $i[$sec];

	# (2) １秒以下の差を算出
	if ( $i[$usec] > $j[$usec] ) {
		$k[$sec]--;
		$j[$usec] = $j[$usec] + 1000000;
	}
	if ( $k[$sec] >= 0 ) {
		$k[$usec] = $j[$usec] - $i[$usec];
	}
	else {
		$k[$usec] = 0;
	}

	# printf "\n** k = [ %d, %d ] ", $k[$sec], $k[$usec];				# ???? 時間差
	return [ @k ];
}

printf("\n(%s) [ %s ] ", $VERSION->[-1], __PACKAGE__ );

#===========================================================
=pod
=encoding utf8
=head1 スクリプト名
Timer.pm - BBS.pm サブモジュール 【 タイマー管理 】
=head1 著者
naoit0 <https://www.naoit0.com/projects/bbs_pm/>
=cut
1;