package BBS::Node;
our $NOLOAD = 1;
#===========================================================
# BBS.pm サブモジュール 【 ノード操作 】
#
our $VERSION = [
	'0001.20250322.2045',		# 初式公開版
];
#===========================================================
# [説明]
#     このモジュールはノードの接続順位と、各ノードが持つパラメータなどのデータを管理します。
#===========================================================
use utf8;
use strict;
use warnings;
use Data::Dumper;
use Debug;

#==================================================
# ●コンストラクタ
# 
# [書式]
#     obj = new()
# 
# [引数]
#     なし
# 
# [返り値]
#     obj =
#         (成功) : オブジェクト
#         (失敗) : undef
# 
# [説明]
#     オブジェクトを作成します。
#==================================================
sub new {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::new';			# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $class = shift;
	my $self = {};
	bless( $self, $class );
	$self->_init();
	return $self;
}

#==================================================
# ●属性の初期化
# 
# [書式]
#     _init()
# 
# [引数]
#     なし
# 
# [返り値]
#     なし
# 
# [説明]
#     本関数は new() より呼び出すもので、直接呼び出し禁止です。
#     オブジェクトの属性を初期化します。
#==================================================
sub _init {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::_init';			# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;
	$self->{'err'}		= 0;			# エラー

	$self->{'Order'}	= [ ];			# ノード開設順
						# [ n, n, ... ]
	$self->{'Node'}		= { };			# ノードテーブル
						# { n }
}

#==================================================
# ●エラー値をセットまたはエラー値を返す
# 
# [書式]
#     err = err();					【 エラー値参照 】
#     err( err );					【 エラー値セット 】
# 
# [引数]
#     err : エラー値				【 セット時 】
# 
# [返り値]
#     err = エラー値				【 参照時 】
# 
# [説明]
#     ルーチン内のエラーをセットまたは参照します。
#==================================================
sub err {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::err';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	return $self->{'err'} if ( $#_ < 0 );			# 引数がなければエラーを返す
	$self->{'err'} = shift;							# 引数があればエラーをセット
}

#==================================================
# ●確保しているノードリストを得る
# 
# [書式]
#     list = nodelist()
# 
# [引数]
#     なし
# 
# [返り値]
#     list = ノードリスト(データ型: 配列)
#           [ num1, num2, ... ]
#           (要素: ノード番号, 順序: 接続順)
# 
# [説明]
#     確保しているノードリストを取得します。
#==================================================
sub nodelist {
	# printf "\n\n=== { %s }", __PACKAGE__.'::nodelist';	# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;

	# my @nodes = @{ $self->{'Order'} };										# ????
	# printf "\n** nodes=(%d)[ %s ]\n", $#nodes+1, join(',', @nodes);			# ???? 接続順リスト

	return @{ $self->{'Order'} };
}

#==================================================
# ●ノードのデータ領域を返す
# 
# [書式]
#     node( num )
#
# [引数]
#     num : ノード番号
#
# [返り値]
#     r =
#         (失敗) : { } 空のハッシュ		【 ※ 注意 ※ 】
#         (成功) : データ領域
#
# [エラーコード]
#     0 : エラーなし(成功)
#     1 : 引数がない
#     2 : 引数が未定義またはノード番号が無効
#     3 : ノードが確保されていない
#
# [説明]
#     ノードのデータ領域を返します。
#
#     各ノードが保持するパラメータなどのデータ領域はノードテーブルで管理しています。
#     ノードテーブルは単純にハッシュ変数ですので、返り値に対してデータを追加、削除することができます。
# 
#    【 ※ 注意 ※ 】
#     エラーによりデータ領域を返せない場合、空のハッシュが返りますが、
#     これは、エラーによるシステム停止を回避するためで、これにより、本関数がエラーにより終了したのか、
#     正常に終了したのかがわかりにくくなっています。
#     エラーを判別する場合は、関数呼び出し後、エラーコードを参照してください。
#==================================================
sub node {
	# printf "\n\n=== { %s }", __PACKAGE__.'::node';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	my $num = shift;				# ノード番号
	$self->err( 0 );				# エラーリセット

	$self->err( 1 ) unless ( defined( $num ) );						# 引数がない(1)
	if ( ( $num !~ /^\d+$/ ) || ( $num < 0 ) ) {
		$self->err( 2 );											# ノード番号が無効(2)
	}
	elsif ( defined( $self->{'Node'}->{$num} ) == 0 ) {
		$self->err( 3 );											# ノードが開いていない(3)
	}
	return { } if ( $self->err() > 0 );								# エラーが発生している

	# printf "\n** node=(%d) ", $num;										# ???? ノード番号
	# print "\n\[Node]\n".Dumper( $self->{'Node'}->{$num} );				# ???? ノード領域の内容

	return $self->{'Node'}->{ $num };
}

#==================================================
# ●ノードの廃棄
#
# [書式]
#     remove( num )
#
# [引数]
#     num : ノード番号
#
# [返り値]
#     r =
#          1 : 廃棄成功
#         -1 : 引数がない
#         -1 : ノード番号が無効
#         -2 : 廃棄失敗
#
# [説明]
#     指定するノードのデータ領域をノードテーブルから削除し、オーダリストから削除します。
#==================================================
sub remove {
	# printf "\n\n=== { %s }", __PACKAGE__.'::remove';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	my $num = shift;				# ノード番号

	# printf "\n** nodes=(%d)[ %s ]\n", $#{ $self->{'Order'} }+1, 
	#				join(',', @{ $self->{'Order'} });					# ???? 接続順リスト（削除前）

	return -1 unless ( defined($num) );									# 引数がない(-1)
	return -1 if ( ($num !~ /^\d+$/) || ($num < 0) );					# ノード番号が無効(-1)

	if ( exists($self->{'Node'}->{$num}) ) {							# ノードが割り当てられている
		# printf "\n** remove=(%d) ", $num;								# ????
		delete( $self->{'Node'}->{$num} );									# ノードテーブルからノードのデータ領域を削除
		$self->del( $num );													# ノード接続順リストから削除
		return 1;															# 削除成功(1)
	}
	else {																# ノードが割り当てられていない
		# print "\n** ( no_delete ) ";										# ????
		return -2;															# 削除失敗(-2)
	}
}

#==================================================
# ●オーダリストの削除
# 
# [書式]
#     del( num )
#
# [引数]
#     num : ノード番号
#
# [返り値]
#     r =
#          1 : 成功
#         -1 : ノード番号が無効
#
# [説明]
#     指定するノード番号をオーダリストから削除します。
#     本関数ではデータ領域がノードテーブルに保持されるので、データ領域を削除するときは必ずremove()を呼び出します。
#==================================================
sub del {
	# printf "\n\n=== { %s }", __PACKAGE__.'::del';			# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	my $num = shift;					# ノード番号

	return -1 unless ( defined($num) );																		# 引数がない(-1)
	return -1 if ( ($num !~ /^\d+$/) || ($num < 0) );														# ノード番号が無効(-1)

	# printf "\n** nodes=(%d)[ %s ] ", $#{ $self->{'Order'} }+1, join(',', @{ $self->{'Order'} });			# ???? 接続順リスト（削除前）
	my @neworder;
	map { push (@neworder, $_) if ( $_ != $num ) } @{ $self->{'Order'} };									# 接続順リストからノード番号を除去
	@{ $self->{'Order'} } = @neworder;
	# printf "\n** nodes=(%d)[ %s ] ", $#{ $self->{'Order'} }+1, join(',', @{ $self->{'Order'} });			# ???? 接続順リスト（削除後）
	return 1;
}

#==================================================
# ●ノードを追加
# 
# [書式]
#     add( num )
#
# [引数]
#     num = ファイル番号(必須)
#
# [返り値]
#     r =
#              -1 : (失敗) 引数がない
#              -1 : (失敗) ファイル番号が無効
#              -2 : (失敗) ノードがすでに割り当てられている
#         ノード番号 : (成功) 追加したノード番号
#
# [説明]
#     新たに接続したノードのデータ領域をノードテーブルに確保します。
#==================================================
sub add {
	# printf "\n\n=== { %s }", __PACKAGE__.'::add';			# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	my $num = shift;														# ファイル番号(必須)

	# printf "\n** num = ( %d ) ", $num if ( defined($num) );					# ???? ノード番号
	# print "\n** ".Dumper( $self->{'Node'} );								# ???? ノードテーブル（追加前）
	# print "\n";															# ????

	return -1 unless ( defined($num) );										# 引数がない(-1)
	return -1 if ( ($num !~ /^\d+$/) || ($num < 0) );						# ファイル番号が無効(-1)
	return -2 if ( exists( $self->{'Node'}->{$num} ) );						# ノードがすでに割り当てられている(-2)

	push( @{ $self->{'Order'} }, $num );									# 接続順リストにノード番号を追加
	$self->{'Node'}->{ $num } = { };										# ノードテーブルにデータ領域を確保

	# print "\n\t**".Dumper( $self->{'Node'} );								# ???? ノードテーブル（追加後）
	# print "\n";															# ????

	return $num;															# ノード番号を返す
}

printf("\n(%s) [ %s ] ", $VERSION->[-1], __PACKAGE__ );

#===========================================================
=pod
=encoding utf8
=head1 スクリプト名
Node.pm - BBS.pm サブモジュール【 ノード操作 】
=head1 著者
naoit0 <https://www.naoit0.com/projects/bbs_pm/>
=cut
1;