package BBS::IO::Output;
our $NOLOAD = 1;
#===========================================================
# BBS.pm サブモジュール 【 出力制御 】
#
our $VERSION = [
	"0001.20250322.2045"			# 正式公開版
];
#===========================================================
# [説明]
#     本モジュールはノードへの送信データ（出力データ）の制御を行います。
# 
#     BBS.pmは通信データのデータ処理は実装されておらず、受け渡ししか行いません。
#     データ処理などはモジュールなど外部の関数で処理することになります。
# 
#     このモジュールではノードへの送信データをどのように処理するのかを示したもので、
#     送信データを受け取るとバッファに取り込み、outputハンドラが呼び出されたときに
#     データを取り出します。
#===========================================================
use utf8;
use strict;
use warnings;
use Encode qw( encode decode );
use Data::Dumper;
use Debug;

use BBS::IO;
use BBS::IO::Buffer;						# バッファ

#==================================================
# ●コンストラクタ
# 
# [書式]
#     obj = new()
# 
# [引数]
#     なし
# 
# [返り値]
#     obj =
#         (成功) : オブジェクト
#         (失敗) : undef
# 
# [説明]
#     オブジェクトを作成します。
#==================================================
sub new {
	# printf "\n\n=== { %s }", __PACKAGE__.'::new';			# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $this = shift;

	my $class = ref($this) || $this;
	my $self = { };
	bless($self, $class);
	$self->_init();
	return $self;
}

#==================================================
# ●属性の初期化
# 
# [書式]
#     _init()
# 
# [引数]
#     なし
# 
# [返り値]
#     なし
# 
# [説明]
#     この関数は new() より呼び出すもので、直接呼び出し禁止です。
#     オブジェクトの属性を初期化します。
#==================================================
sub _init {
	# printf "\n\n=== { %s }", __PACKAGE__.'::_init';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;

	# エラーコード
	$self->{'err'}			= 0;

	# バッファ
	$self->{'__Buffer'}		= new BBS::IO::Buffer();
}

#==================================================
# ●エラー値をセットまたはエラー値を返す
# 
# [書式]
#     err = err();					【 エラー値参照 】
#     err( err );					【 エラー値セット 】
# 
# [引数]
#     err : エラー値				【 セット時 】
# 
# [返り値]
#     err = エラー値				【 参照時 】
# 
# [説明]
#     ルーチン内のエラーをセットまたは参照します。
#==================================================
sub err {
	# printf "\n\n=== { %s }", __PACKAGE__.'::err';			# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;

	return $self->{'err'} if ( $#_ < 0 );			# 引数がなければエラーを返す
	$self->{'err'} = shift;							# 引数があればエラーをセット
}

#==================================================
# ●送出データを送信バッファの後方から取り込む
# 
# [書式]
#     store( data )
# 
# [引数]
#     data = データブロック(データ型: 配列)
# 
# [返り値]
#     なし
# 
# [説明]
#     送出データを送信バッファの後方から取り込みます。
#     取り込まれるデータは、すでに取り込まているデータが送出された後で送出されます。
# 
# [メモ]
#==================================================
sub store {
	# printf "\n\n=== { %s }", __PACKAGE__.'::store';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	$self->{'__Buffer'}->lastin( @_ );
}

#==================================================
# ●送出データを送信バッファの前方から取り込む
# 
# [書式]
#     insert( data )
# 
# [引数]
#     data = データブロック(データ型: 配列)
# 
# [返り値]
#     なし
# 
# [説明]
#     送出データを送信バッファに取り込みます。
#     本関数で取り込まれるデータは、それまでに取り込まれているデータの前に優先して送出されます。
# 
# [メモ]
#      
#==================================================
sub insert {
	# printf "\n\n=== { %s }", __PACKAGE__.'::insert';			# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;
	$self->{'__Buffer'}->firstin( @_ );
}

#==================================================
# ●送信バッファの送信データを参照する
# 
# [書式]
#     data = get( num )
# 
# [引数]
#     num : レコード番号( 省略時: 0 )
# 
# [返り値]
#     data = 参照したデータ
# 
# 
# [説明]
#     送信バッファから送出データを参照します。
# 
# [メモ]
#     この関数ではバッファからデータを抜き出しませんので、データはバッファに残ります。
#     データの抜き出しは関数 fetch() を使用してください。
#==================================================
sub get {
	# printf "\n\n=== { %s }", __PACKAGE__.'::get';			# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	my $num = shift || 0;					# レコード番号
	my $buf = $self->{'__Buffer'};

	my $r = $buf->get( $num, 1 );			# 指定値レコード番号のブロックを取得
	$self->err( $buf->err() );				# Bufferのエラーを回収
	return $r;
}

#==================================================
# ●送信バッファから送信データを抜き出す
# 
# [書式]
#     data = fetch()
# 
# [引数]
#     なし
# 
# [返り値]
#     data = 取り出したデータ
# 
# [説明]
#     送信バッファから送出データを抜き出します。
# 
# [メモ]
#     この関数ではバッファからデータを抜き出します。
#     データの参照は関数 get() を使用してください。
#==================================================
sub fetch {
	# printf "\n\n=== { %s }", __PACKAGE__.'::fetch';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;

	my $buf = $self->{'__Buffer'};
	my $r = $buf->firstout( 1 );			# 先頭のブロックを取り出す
	$self->err( $buf->err() );				# Bufferのエラーを回収
	return $r;
}

#==================================================
# ●送信バッファに保存されているデータ（ブロック）数を返す
# 
# [書式]
#     cnt = count()
# 
# [引数]
#     なし
# 
# [返り値]
#     cnt = データ（ブロック）の総数
# 
# [説明]
#    送信バッファに保存されているデータ（ブロック）数を返します。
#==================================================
sub count {
	# printf "\n\n=== { %s }", __PACKAGE__.'::count';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	return $self->{'__Buffer'}->count();
}

#==================================================
# ●送信バッファをクリア
# 
# [書式]
#     clear()
# 
# [引数]
#     なし
# 
# [返り値]
#     なし
# 
# [説明]
#     送信バッファをクリアします。
#==================================================
sub clear {
	# printf "\n\n=== { %s }", __PACKAGE__.'::clear';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	return $self->{'__Buffer'}->clear();				
}

printf("\n(%s) [ %s ] ", $VERSION->[-1], __PACKAGE__ );

#===========================================================
=pod
=encoding utf8
=head1 スクリプト名
Output.pm - BBS.pm サブモジュール 【 出力制御 】
=head1 著者
naoit0 <https://www.naoit0.com/projects/bbs_pm/>
=cut
1;