package BBS::IO::Input;
our $NOLOAD = 1;
#===========================================================
# BBS.pm サブモジュール 【 入力制御 】
#
our $VERSION = [
	"0001.20250322.2045"			# 正式公開版
];
#===========================================================
# [説明]
#     本モジュールはノードからの受信データ（入力データ）の制御を行います。
# 
#     BBS.pmは通信データのデータ処理は実装されておらず、受け渡ししか行いません。
#     データ処理などはモジュールなど外部の関数で処理することになります。
# 
#     このモジュールではノードからの受信データをどのように処理するのかを示したもので、
#     受信データを受け取るとバッファに取り込み、取り出しの際はデータ型を指定して取り出します。
#===========================================================
use utf8;
use strict;
use warnings;
use Encode qw( encode decode );
use Data::Dumper;
use Debug;

use BBS::IO;								# バッファ
use BBS::IO::Buffer;						# バッファ

#==================================================
# ●コンストラクタ
# 
# [書式]
#     obj = new()
# 
# [引数]
#     なし
# 
# [返り値]
#     obj =
#         (成功) : オブジェクト
#         (失敗) : undef
# 
# [説明]
#     オブジェクトを作成します。
#==================================================
sub new {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::new';			# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $this = shift;

	my $class = ref($this) || $this;
	my $self = { };
	bless($self, $class);
	$self->_init();
	return $self;
}

#==================================================
# ●属性の初期化
# 
# [書式]
#     _init()
# 
# [引数]
#     なし
# 
# [返り値]
#     なし
# 
# [説明]
#     この関数は new() より呼び出すもので、直接呼び出し禁止です。
#     オブジェクトの属性を初期化します。
#==================================================
sub _init {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::_init';			# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;

	# エラーコード
	$self->{'err'}			= 0;

	# 受信バッファ
	$self->{'__Buffer'}		= new BBS::IO::Buffer();
}

#==================================================
# ●エラー値をセットまたはエラー値を返す
# 
# [書式]
#     err = err();					【 エラー値参照 】
#     err( err );					【 エラー値セット 】
# 
# [引数]
#     err : エラー値				【 セット時 】
# 
# [返り値]
#     err = エラー値				【 参照時 】
# 
# [説明]
#     ルーチン内のエラーをセットまたは参照します。
#==================================================
sub err {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::err';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;
	return $self->{'err'} if ( $#_ < 0 );			# 引数がなければエラーを返す
	$self->{'err'} = shift;							# 引数があればエラーをセット
}

#==================================================
# ●受信データを受信バッファの後方から取り込む
# 
# [書式]
#     store( data )
# 
# [引数]
#    data = 受信データ
# 
# [返り値]
#    なし
#
# [エラーコード]
#     1 : 受信データが未定義またはヌル
#     2 : 引数が無効
# 
# [説明]
#     受信データを受信バッファの後方から取り込みます。
#     取り込むデータは１バイトずつ分割されます。
#     取り込むデータは必ず utf8 でデコードする必要があります。
#
# [メモ]
#==================================================
sub store {
	# printf "\n\n=== { %s }", __PACKAGE__.'::store';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	my $data = shift;

	$self->err( 0 );											# エラーリセット
	$self->err( 2 ) unless ( ref( $data ) eq '' );				# 引数の型が無効(2)
	return if ( $self->err() > 0 );

	if ( ( defined( $data ) == 1 ) && ( $data ne '' ) ) {
		# print "\n( ok )";										# ????
		my @bytes_data = split( //, $data );					# データを１バイトごとに分割
		$self->{'__Buffer'}->lastin( @bytes_data );				# 受信バッファの後方から取り込む
	}
	else {
		# print "\n( no )";										# ????
		$self->err( 1 );										# 受信データが未定義またはヌル(1)
	}
}

#==================================================
# ●受信データを受信バッファの前方から取り込む
# 
# [書式]
#     insert( data )
# 
# [引数]
#    data = 受信データ
# 
# [返り値]
#    なし
#
# [エラーコード]
#     1 : 受信データが未定義またはヌル
#     2 : 引数が無効
# 
# [説明]
#     受信データを受信バッファの前方から取り込みます。
#     取り込むデータは１バイトずつ分割されます。
#     取り込むデータは必ず utf8 でデコードする必要があります。
#
# [メモ]
#==================================================
sub insert {
	# printf "\n\n=== { %s }", __PACKAGE__.'::insert';	# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	my $data = shift;

	$self->err( 0 );											# エラーリセット
	$self->err( 2 ) unless ( ref( $data ) eq '' );				# 引数の型が無効(2)
	return if ( $self->err() > 0 );

	if ( ( defined( $data ) == 1 ) && ( $data ne '' ) ) {
		# print "\n( ok )";										# ????
		my @bytes_data = split( //, $data );					# データを１バイトごとに分割
		$self->{'__Buffer'}->firstin( @bytes_data );			# 受信バッファの先頭から取り込む
		# $self->__t_buffer();									# バッファ検証
	}
	else {
		# print "\n( no )";										# ????
		$self->err( 1 );										# 受信データが未定義またはヌル(1)
	}
}

#==================================================
# ●受信バッファのデータを参照する
# 
# [書式]
#     data = get( offset, length )		# 指定する開始位置から指定するデータ長までを取得
#     data = get( length )				# バッファ前方から指定するデータ長までを取得
#     data = get()						# 全てのデータを取得
# 
# [引数]
#    offset : 開始位置
#    length : データ長(バイト数)
# 
# [返り値]
#    data : データ
#
# [エラーコード]
#     なし
# 
# [説明]
#     受信バッファに保存されているデータを参照します。
#
# [メモ]
#     この関数はバッファモジュールの関数 get() のエイリアスです。
#     この関数ではデータの参照のみでデータの取り出しは行いません。
#==================================================
sub get {
	printf "\n\n=== { %s }", __PACKAGE__.'::get';		# ????
	printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	my $buf = $self->{'__Buffer'};
	my $r = $buf->get( @_ );
	$self->err( $buf->err() );
	return $r;
}

#==================================================
# ●受信バッファの先頭から一文字参照する
# 
# [書式]
#     [ len, chr ] = getchr()
# 
# [引数]
#    なし
# 
# [返り値]
#    (成功) : [ len: データ長, chr: 文字 ]
#    (失敗) : undef
#
# [エラーコード]
#     なし
# 
# [説明]
#     受信バッファの先頭から一文字分、参照します。
#
# [メモ]
#     この関数ではデータの参照のみでデータの取り出しは行いません。
#     データの取り出しは関数 firstout() を呼び出します。
#==================================================
sub getchr {
	printf "\n\n=== { %s }", __PACKAGE__.'::getchr';	# ????
	printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;

	my @datas = @{ $self->get( 4 ) };			# バッファの先頭から4バイト取得
	return if ( $#datas == -1 );				# データなし

	print "\n{ datas }\n";
	$self->BBS::IO::_dmp( join( '', @datas ) );			# ????

	# (2) 文字識別
	my $chr = '';											# 文字データ
	my $r = 0;												# データ長
	until ( $r != 0 ) {
		$chr .= shift( @datas );							# 連結
		$r = $self->BBS::IO::is_utf8_p2( $chr );				# 識別
		if ( ( $r == 0 ) && ( $#datas < 0 ) ) {
			$r = -1; $chr = undef;
		}
	}

	# printf "\nlen = ( %s )", $r;						# ????
	# print "\n\t\t\t\t{ chr }";
	$self->BBS::IO::_dmp( $chr,4 );		# ????

	return [ $r, $chr ];
}

#==================================================
# ●受信バッファの先頭から一文字取り出す
# 
# [書式]
#     chr_utf8dec = fetchchr()
# 
# [引数]
#    なし
# 
# [返り値]
#    chr_utf8dec : 取り出した文字（utf8デコード）
#
# [エラーコード]
#     1 : 受信バッファが空
#     2 : utf8文字の取得に失敗した
# 
# [説明]
#     受信バッファに保存されているデータを先頭から取り出します。
#
# [メモ]
#     この関数ではデータの取り出しを行うと、バッファからデータが消去されます。
#     バッファのデータを保持したまま参照するには関数 get() を呼び出します。
#==================================================
sub fetchchr {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::fetchchr';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;
	$self->err( 0 );			# エラーリセット

	my $cnt = $self->count();				# (1) 受信バッファのデータ長を取得
	# printf "\n*** count = ( %d )", $cnt;	# ????
	if ( $cnt < 1 ) {
		$self->err( 1 );					# 受信バッファが空(1)
		return;
	}

	my ( $r, $chr ) = @{ $self->getchr() };							# 一文字参照する
	# print "\n{ chr }";					# ????
	# $self->BBS::IO::_dmp( $chr );		#
	# printf "\nr = ( %d )", $r;			# ????

	$r = 1 if ( $r <= 0 );										# 文字の参照に失敗→受信バッファから先頭１バイト分を取り出す
	$chr = $self->fetch( $r );									# データを抜き出す
	$chr = join( '', @{ $chr } );
	$chr = decode( 'utf8', $chr );

	return $chr;
}

#==================================================
# ●受信バッファのデータを取り出す
# 
# [書式]
#     data = fetch( size )
# 
# [引数]
#    size : データサイズ
# 
# [返り値]
#    data : データ
#
# 「エラーコード]
#     1 : 引数が無効
#     2 : データがない
#     3 : 取得したブロック数が指定値より少ない
# 
# [説明]
#     受信バッファに保存されているデータを先頭から取り出します。
#
# [メモ]
#     この関数はバッファモジュールの関数 firstout() のエイリアスです。
#==================================================
sub fetch {
	# printf "\n\n=== { %s }", __PACKAGE__.'::fetch';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	my $buf = $self->{'__Buffer'};
	my $r = $buf->firstout( @_ );
	$self->err( $buf->err() );
	return $r;
}

#==================================================
# ●受信バッファのデータ長を返す
# 
# [書式]
#     cnt = count()
# 
# [引数]
#    なし
# 
# [返り値]
#    cnt : データ長
#
# [エラーコード]
#     なし
# 
# [説明]
#     受信バッファに取り込魔れているデータ長（サイズ）を返します。
#
# [メモ]
#==================================================
sub count {
	# printf "\n\n=== { %s }", __PACKAGE__.'::count';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	return $self->{'__Buffer'}->count();
}

printf("\n(%s) [ %s ] ", $VERSION->[-1], __PACKAGE__ );

#===========================================================
=pod
=encoding utf8
=head1 スクリプト名
Input.pm - BBS.pm サブモジュール 【 入力制御 】
=head1 著者
naoit0 <https://www.naoit0.com/projects/bbs_pm/>
=cut
1;