package BBS::IO::Buffer;
our $NOLOAD = 1;
#===========================================================
# BBS.pm サブモジュール 【 汎用バッファ 】
# 
our $VERSION = [
	"0001.20250322.2045", 		# 正式公開版
];
#===========================================================
# [説明]
#     本モジュールは汎用バッファを作成します。
#     バッファのデータはブロック単位で保存します。
# 
#      firstin()  ->  +------------------------+  <- lastin()
#                     | (First) | ... | (Last) |
#      firstout() <-  +------------------------+  -> lastout()
#===========================================================
use utf8;
use strict;
use warnings;
use Encode qw( encode decode );
use Data::Dumper;
use Debug;

#==================================================
# ●コンストラクタ
# 
# [書式]
#     obj = new()
# 
# [引数]
#     なし
# 
# [返り値]
#     obj =
#         (成功) : オブジェクト
#         (失敗) : undef
# 
# [説明]
#     オブジェクトを作成します。
#==================================================
sub new {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::new';			# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $class = shift;
	my $self = { @_ };
	bless($self, $class);
	$self->_init();
	return $self;
}

#==================================================
# ●属性の初期化
# 
# [書式]
#     r = _init()
# 
# [引数]
#     なし
#
# [返り値]
#     なし
#
# [説明]
#     この関数は new() より呼び出すもので、直接呼び出し禁止です。
#     オブジェクトの属性を初期化します。
#==================================================
sub _init {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::_init';			# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;

	$self->{'err'}			= 0;		# エラー
	$self->{'Buffer'}		= [ ];		# バッファ
}

#==================================================
# ●エラー値をセットまたはエラー値を返す
# 
# [書式]
#     err = err();					【 エラー値参照 】
#     err( err );					【 エラー値セット 】
# 
# [引数]
#     err : エラー値				【 セット時 】
# 
# [返り値]
#     err = エラー値				【 参照時 】
# 
# [説明]
#     ルーチン内のエラーをセットまたは参照します。
#==================================================
sub err {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::err';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	return $self->{'err'} if ( $#_ < 0 );			# 引数がなければエラーを返す
	$self->{'err'} = shift;							# 引数があればエラーをセット
}

#==================================================
# ●バッファの前方からブロックを追加する
#
# [書式]
#     firstin( block )
# 
# [引数]
#     block : ブロック(データ型: 配列)
# 
# [返り値]
#     なし
# 
# [説明]
#     配列操作の unshift() と同じ動作をします。
#==================================================
sub firstin {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::firstin';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;
	unshift( @{ $self->{'Buffer'} }, @_ );						# バッファに取り込む
	return;
}

#==================================================
# ●バッファの後方からブロックを追加する
#
# [書式]
#     lastin( block )
# 
# [引数]
#     block : ブロック(データ型: 配列)
# 
# [返り値]
#     なし
# 
# [説明]
#     配列操作の push() と同じ動作をします。
#==================================================
sub lastin {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::lastin';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	push( @{ $self->{'Buffer'} }, @_ );							# バッファに取り込む
	return;
}

#==================================================
# ●バッファの前方からブロックを抜き出す
#
# [書式]
#     block = firstout( length )
# 
# [引数]
#     length : 取り出すブロック数(省略時: 1)
# 
# [返り値]
#     block =
#         (失敗) : undef
#         (成功) : 抜き出したブロック(データ型: 配列リファレンス)
# 
# 「エラーコード]
#     1 : 引数が無効
#     2 : データがない
#     3 : 取得したブロック数が指定値より少ない
# 
# [説明]
#     配列操作の shift() と同じ動作をします。
#     指定するブロック数がバッファに保存されているブロック数を超えた場合、全てのブロックを抜き出します。
#==================================================
sub firstout {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::firstout';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;
	$self->err( 0 );		# エラーリセット

	my $length = shift;												# 取り出すブロック数
		$length = defined($length) ? $length : 1;
	return if ( ( $length !~ /^\d+$/ ) || ( $length == 0 ) );		# 引数が数値でないかゼロ(undef)

	if ( $self->count() <= 0 ) {									# ブロックが空(2)
		$self->err(2);
		return;
	}

	# printf "\n** length = ( %d )", $length;						# ???? 取り出すブロック数
	# printf "\n** buffer.count = ( %d )\n", $self->count();		# ???? バッファの総ブロック数

 	my @blocks = splice( @{ $self->{'Buffer'} }, 0, $length );		# 前方取得
	$self->err(3) if ( $length > $#blocks+1 );						# 取得したブロック数が引数より少ない(3)

	return [ @blocks ];
}

#==================================================
# ●バッファの後方からブロックを抜き出す
#
# [書式]
#     data = lastout( length )
# 
# [引数]
#     length : 取り出すブロック数(省略時: 1)
# 
# [返り値]
#     data = 
#         (失敗) : undef
#         (成功) : 抜き出したブロック(データ型: 配列リファレンス)
#                 [ b1, b2, ... ]
# 
# 「エラーコード]
#     1 : 引数が無効
#     2 : ブロックが空
# 
# [説明]
#     バッファの後方からブロックを抜き出します。
#     配列操作の pop() と同じ動作をします。
#     指定するブロック数がバッファに保存されているブロック数を超えた場合、全てのブロックを抜き出します。
#==================================================
sub lastout {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::lastout';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;
	$self->err( 0 );		# エラーリセット

	my $length = shift;												# 取り出すブロック数
		$length = defined($length) ? $length : 1;
	return if ( ( $length !~ /^\d+$/ ) || ( $length <= 0 ) );		# 引数が数値でないかゼロ以下(undef)

	if ( $self->count() < 0 ) {
		$self->err(2);												# ブロックが空(2)
		return;
	}

	# printf "\n** length = ( %d )", $length;						# ???? 取り出すブロック数
	# printf "\n** buffer.count = ( %d )\n", $self->count();		# ???? バッファの総ブロック数

	my @blocks;
	if ( $self->count() > $length ) {								# ブロック数が指定値以上（後方取得）
		$length = $length - ( $length * 2 );
		@blocks = splice( @{ $self->{'Buffer'} }, $length );
	}
	else {															# ブロック数が指定値以下（全て取得）
		@blocks = splice( @{ $self->{'Buffer'} }, 0, $length );
	}
	return [ @blocks ];
}

#==================================================
# ●バッファの内容を参照する
#
# [書式]
#     data = get( offset, length )		# 指定するブロックから指定するブロック数までを取得
#     data = get( length )				# 先頭ブロックから指定するブロック数までを取得
#     data = get()						# 先頭ブロックのみを取得
#     data = get( -1 )					# 全てのブロックを取得
# 
# [引数]
#     offset = 取得するデータ（ブロック）の起点
#     length = 取得するデータ（ブロック）数
# 
# 「エラーコード]
#     1 : 引数が無効
#
# [返り値]
#     data = 取得したブロック(データ型: 配列リファレンス)
#            [ b1, b2, ... ]
#
# [説明]
#     バッファのデータ（ブロック）を参照します。
#     データ（ブロック）の抜き出しは行いません。
#==================================================
sub get {
	printf "\n\n=== { %s } ", __PACKAGE__.'::get';		# ????
	printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	my @blocks = @{ $self->{'Buffer'} };	# バッファ（コピー）

	my ( $offset, $length );				# 
	$self->err( 0 );						# エラーリセット

	# ● 引数なし :  ( )
	if ( $#_ == -1 ) {						# ● 引数なし
		return [ shift( @blocks ) ];	 		# → 前方ブロックのみ返す
	}

	# ● 引数が１つ : ( length )
	elsif ( $#_ == 0 ) {
		$length = shift || 0;
		if ( $length == -1 ) {				# ● -1
			return [ @blocks ];					# → 全データを返す
		}
		elsif ( $length > 0 ) {				# ● 0以上
			$offset = 0;						# → 前方ブロックから指定値のブロックまで返す
		}
		else {								# ● それ以外
			$self->err( 1 );					# → 引数が無効(1)
		}
	}

	# ● 引数が２つ : ( offset, length )
	elsif ( $#_ == 1 ) {					# ● 引数が２つ
		$offset = shift || 0;					# 指定位置から
		$length = shift || 0;					# 指定長分のブロックを返す
		if ( $offset > $#blocks ) {
			$self->err( 4 );					# → 引数が無効(4)
		}
	}

	$self->err( 2 ) unless ( defined($offset) );				# → splice()に渡す引数が無効(2)
	$self->err( 3 ) unless ( defined($offset) );				# → splice()に渡す引数が無効(3)
	return if ( $self->err() > 0 );								# エラーなら終了

	# printf "\n offset = ( %d )", $offset;						# ????
	# printf "\n length = ( %d )", $length;						# ????
	my @trim_blocks = splice( @blocks, $offset, $length );
	# print "\n== { trim_brocks }\n".Dumper( @trim_blocks );	# ????
	return [ @trim_blocks ];
}

#==================================================
# ●バッファに保存されているブロック数を返す
# 
# [書式]
#     cnt = count()
# 
# [引数]
#     なし
# 
# [返り値]
#     cnt = ブロック数(0: ブロックが無い)
# 
# [説明]
#     
#==================================================
sub count {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::count';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	return ( $#{ $self->{'Buffer'} } + 1 );
}

#==================================================
# ●バッファをクリアする
# 
# [書式]
#     clear()
# 
# [引数]
#     なし
# 
# [返り値]
#     なし
#==================================================
sub clear {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::clear';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	return $self->_init();
}

printf("\n(%s) [ %s ] ", $VERSION->[-1], __PACKAGE__ );

#===========================================================
=pod
=encoding utf8
=head1 スクリプト名
Buffer.pm - BBS.pm サブモジュール 【 汎用バッファ 】
=head1 著者
naoit0 <https://www.naoit0.com/projects/bbs_pm/>
=cut
1;