package BBS::IO;
our $NOLOAD = 1;
#===========================================================
# BBS.pm サブモジュール 【 入出力制御（共通関数） 】
#
our $VERSION = [
	"0001.20250322.2045"			# 正式公開版
];
#===========================================================
# [説明]
#     このモジュールは配下モジュールの共通関数を収録したものです。
#===========================================================
use utf8;
use strict;
use warnings;
use Encode qw( encode decode );
use Data::Dumper;
use Debug;

#==================================================
# ●コンストラクタ
# 
# [書式]
#     obj = new()
# 
# [引数]
#     なし
# 
# [返り値]
#     obj =
#         (成功) : オブジェクト
#         (失敗) : undef
# 
# [説明]
#     オブジェクトを作成します。
#==================================================
sub new {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::new';			# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $this = shift;

	my $class = ref($this) || $this;
	my $self = { };
	bless($self, $class);
	$self->_init();
	return $self;
}

#==================================================
# ●属性の初期化
# 
# [書式]
#     _init()
# 
# [引数]
#     なし
# 
# [返り値]
#     なし
# 
# [説明]
#     この関数は new() より呼び出すもので、直接呼び出し禁止です。
#     オブジェクトの属性を初期化します。
#==================================================
sub _init {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::_init';			# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;

	# エラーコード
	$self->{'err'}			= 0;
}

#==================================================
# ●エラー値をセットまたはエラー値を返す
# 
# [書式]
#     err = err();					【 エラー値参照 】
#     err( err );					【 エラー値セット 】
# 
# [引数]
#     err : エラー値				【 セット時 】
# 
# [返り値]
#     err = エラー値				【 参照時 】
# 
# [説明]
#     ルーチン内のエラーをセットまたは参照します。
#==================================================
sub err {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::err';			# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;
	return $self->{'err'} if ( $#_ < 0 );			# 引数がなければエラーを返す
	$self->{'err'} = shift;							# 引数があればエラーをセット
}

#==================================================
# ●文字列がエンコードまたはデコードされているかどうか確認
# 
# [書式]
#     r = is_decode( str );
#     r = is_encode( str );
# 
# [引数]
#    str : 文字列
# 
# [返り値]
#    r =
#        0: エンコードされている, 1: デコードされている		( is_decode() 呼び出し時 )
#        0: デコードされている, 1: エンコードされている		( is_encode() 呼び出し時 )
# 
# [エラーコード]
#     なし
# 
# [説明]
#     文字列がエンコードまたはデコードされているかどうかチェックします
# 
# [メモ]
#     デコードしたデータは文字単位で管理されているため、一文字の文字コードが255を超えるものがあり、
#     エンコードしたものはバイト単位で管理されているため、１バイトの上限は255なので、
#     255を超えることはありません。
#     この関数では、すべてのデータを取り出し、255を超えた場合はデコードデータ、
#     255を超えなかった場合はエンコードデータとして判定してします。
#     関数 is_encode() は、本関数の返り値を反転したものを返しています。
#==================================================
sub is_decode {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::is_decode';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;
	my $data = shift;
	my $r = 0;

	foreach my $chr ( split( //, $data ) ) {
	 	my $oct_chr = ord( $chr );
		# printf "\n [%02x] (%d)", $oct_chr, $oct_chr;		# ????
		if ( $oct_chr > 255 ) {								# 文字コードが255を超えた
			$r = 1; last;										# デコード確定(1)
		}
	}
	# printf "\n\t is_decode.r = ( %d ）", $r;				# ????
	return $r;
}

sub is_encode {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::is_encode';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;

	my $r = ( $self->is_decode( @_ ) == 1 ) ? 0 : 1;
	# printf "\n\t is_encode.r = ( %d ）", $r;				# ????
	return $r;
}

#==================================================
# ●データがＵＴＦ８であるか調べる
# 
# [書式]
#     r = is_utf8( str_utf8enc )
# 
# [引数]
#    str_utf8enc : ＵＴＦ８文字列（エンコード）
# 
# [返り値]
#    r : 識別結果()
#         0: ＵＴＦ８でない（失敗）
#         1: ＵＴＦ８である（成功）
#
# [エラーコード]
#     なし
# 
# [説明]
#     データがＵＴＦ８であるか調べます。
#
# [メモ]
#     この関数では、受け取ったデータをいったんエンコードしてバイナリ単位取り出せるようにし、
#     １バイトずつ取り出しながら評価します。
#     まず、ＵＴＦ８文字の先頭データを見つけ、そのデータから後半のデータ長分、取り出して
#     先頭データと結合します。
#     結合したデータがＵＴＦ８に該当しない場合、その時点で失敗と判定し、処理を中断します。
#==================================================
sub is_utf8 {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::is_utf8';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	my $data = shift;		# データ（utf8エンコード）

	# (1) データタイプがデコードならエンコードする
	$data = encode( 'utf8', $data ) if ( $self->is_decode( $data ) );
	$data = [ split( //, $data ) ];

 	my $exit = 0;
 	until ( $exit == 1 ) {

		# (1) １バイト取り出し、取り出したデータに応じて、取り出す後部データのデータ長を決定
 		my $l = 0;										# 取得バイト数
 		my $chr = shift( @{ $data } );					# 先頭の１バイト取り出し
		if ( $chr =~ /^[\xC2-\xDF]/ ) {					# ２バイト文字の先頭
			$l = 1;											# １バイト取り出し
		}
		elsif ( $chr =~ /^[\xE0-\xEF]/ ) {				# ３バイト文字の先頭
			$l = 2;											# ２バイト取り出し
		}
		elsif ( $chr =~ /^[\xF0-\xF4]/ ) {				# ４バイト文字の先頭
			$l = 3;											# ３バイト取り出し
		}
		# printf "\nl = ( %d )", $l;							# ????	# 後部データのバイト数

 		if ( ( $l >= 1 ) && ( $l <= 3 ) ) {
			$chr .= join( '', splice( @{ $data }, 0, $l ) );	# データ取り出し、先頭データと結合
			# print "\n\n{ chr }"; $self->_dmp( $chr, 0 );		# ????
		}

		# (2) 結合したデータがutf8文字に該当するか判定
 		my $r = 0;										# 文字判定フラグ
		$r = $self->is_utf8_p2( $chr );
		# printf "\nr = ( %d )\n", $r;						# ???? 判定値

		# (3) 結果処理
		if ( $r == 0 ) {					# 識別失敗
			# print "  - error";				# ????
			# print "\n * { about }";			# ????
			return 0;						# 識別失敗(0)
		}
 		elsif ( $#{ $data } < 0 ) {			# 全てのデータを調べた
			$exit = 1;							# ループ終了
		}
 	}
	# print "\n ";
	return 1;							# 識別成功(1)
}

sub is_utf8_p2 {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::is_utf8_p2';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;
	my $chr = shift;		# データ（utf8エンコード）
	my $r = 0;										# 文字判定フラグ

	# データがutf8文字に該当するか判定

	if ( $chr =~ /^[\x00-\x7f]$/ ) {										# １バイト文字( 制御文字・ASCII )
		$r = 1;
	}
	elsif ( $chr =~ /^[\xC2-\xDF][\x80-\xBF]$/ ) {							# ２バイト文字
		$r = 2;
	}
	elsif ( $chr =~ /^[\xE0-\xEF][\x80-\xBF][\x80-\xBF]$/ ) {				# ３バイト文字
		$r = 3;
	}
	elsif ( $chr =~ /^[\xF0-\xF4][\x80-\xBF][\x80-\xBF][\x80-\xBF]$/ ) {	# ４バイト文字
		$r = 4;
	}
	# printf "\nr = ( %d )\n", $r;						# ???? 判定値
	return $r;
}

#==================================================
# ●ＵＴＦ８文字列からＣＰ９３２文字列でのデータ長を計算する
# 
# [書式]
#     len = cp932len( data )
# 
# [引数]
#    data : ＵＴＦ８文字列(デコード)
# 
# [返り値]
#    len : (失敗) : -1
#          (成功) : ＣＰ９３２文字データ長

#
# [エラーコード]
#     なし
# 
# [説明]
# 
# [メモ]
#     ＵＴＦ８文字列からＣＰ９３２文字列でのデータ長を計算します。
#     ASCIIなど、半角文字はＣＰ９３２もＵＴＦ８も同じ１バイトですが、
#     漢字やひらがななど、全角文字はＣＰ９３２では２バイト、
#     ＵＴＦ８では全角文字は２～４バイト（おおむね３バイト）で構成しています。
#     半角カタカナの場合、ＵＴＦ８は３バイトですが、
#     ＣＰ９３２は１バイトになるので、この関数では、その部分を補正して算出しています。
#     本関数が必要である理由は、コンソールに出力したときの文字とデータ長をそろえる必要があるためです。
#==================================================
sub cp932len {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::cp932len';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	my $data = shift;


 	# (1) データの文字コードがＵＴＦ８か判別
	my $is_utf8 = $self->is_utf8( $data );
	# printf "\n * is_utf8 = ( %d )", $is_utf8;			# ???? ＵＴＦ８判定値
	return -1 if ( $is_utf8 == 0 );						# →文字コードがＵＴＦ８でない(-1)

	# (2) データタイプがエンコードならデコードする
	$data = decode( 'utf8', $data ) if ( $self->is_encode( $data ) );
	# print "\n{ data }\n"; $self->_dmp( $data, 2 );							# ????
	$data = [ split( //, $data ) ];

	# (3) ＣＰ９３２文字長を算出
	my $len_total = 0;
	map {
		my $chr = $_;
		my $len = 0;

		$chr = encode( 'utf8', $chr );			# 文字をエンコード
		# $self->_dmp( $chr, 5 );					# ????
		if ( $chr =~ /^[\x00-\x7f]$/ ) {										# 制御文字・ASCII(1バイト)
			$len = 1;
		}
		elsif ( $chr =~ /^\xEF(\xBD[\xA1-\xBF]|\xBE[\x80-\x9F])$/ ) {			# 半角カタカナ(1バイト)
			$len = 1;
		}
		else {																	# その他（漢字など）(2バイト)
			$len = 2;
		}

		$len_total = $len_total + $len;
		# printf "\n\n cp932len = ( %d )", $len;				# ????
		# printf ", total = ( %d )\n", $len_total;			# ????
	} @{ $data };

	return $len_total;
}

#==================================================
# ●データをダンプします。
# 
# [書式]
#     _dmp( data, indent )
# 
# [引数]
#      data : データ( バイトデータ列挙 : "\xnn\xnn\xnn..." );
#	 indent : インデント桁数
# 
# [返り値]
#    なし
#
# [エラーコード]
#     なし
# 
# [説明]
#     データをダンプします。
#     本関数はデータ検証用です。
#     データをＵＴＦ８文字としてダンプする場合、あらかじめデコードしたものを指定します。
#     データをバイナリとしてダンプする場合、あらかじめエンコードしたものを指定します。
#
# [メモ]
#==================================================
sub _dmp {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::_dmp';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	my $data = shift;
	my $indent = shift;
		$indent = ( defined( $indent ) ) ? "\t" x $indent : "\t";

	# printf "\ndata.len = ( %d )", length( $data );		# ????
	map {
		printf "\n%s [%02x] (%d)", $indent, ord( $_ ), ord( $_ );
	} split( //, $data );
	print "\n";
}


printf("\n(%s) [ %s ] ", $VERSION->[-1], __PACKAGE__ );

#===========================================================
=pod
=encoding utf8
=head1 スクリプト名
IO.pm - BBS.pm サブモジュール 【 入出力制御（共通関数） 】
=head1 著者
naoit0 <https://www.naoit0.com/projects/bbs_pm/>
=cut
1;