package BBS::Handler;
our $NOLOAD = 1;
#===========================================================
# BBS.pm サブモジュール 【 ハンドラ制御 】
#
our $VERSION = [
	'0001.20250322.2045',			# 正式公開版
];
#===========================================================
# [説明]
#     このモジュールは外部ルーチンを呼び出すハンドラを作成します。
#     オブジェクト１つに複数のルーチンを設定または呼び出すことができます。
#     ルーチンは登録した順に呼び出しますが、順序の変更は関数 order() で行えるほか、
#      set() で登録済みのルーチンに挿入して登録することもできます。
#     登録したルーチンを呼び出すとき、引数を与えることができません。
#     詳細についてはドキュメントをご覧ください。
#===========================================================
use utf8;
use strict;
use warnings;
use Scalar::Util qw( refaddr );
use Data::Dumper;
use Debug;

#==================================================
# ●コンストラクタ
# 
# [書式]
#     obj = new()
# 
# [引数]
#     なし
# 
# [返り値]
#     obj =
#         (成功) : オブジェクト
#         (失敗) : undef
# 
# [説明]
#     オブジェクトを作成します。
#     引数を省略した場合はオブジェクトの作成のみを行います。
#==================================================
sub new {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::new';			# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $this = shift;
	my $class = ref($this) || $this;
	my $self = {};
	bless( $self, $class );
	$self->_init();
	return $self;
}

#==================================================
# ●属性の初期化
# 
# [書式]
#     r = _init()
# 
# [引数]
#     なし
#
# [返り値]
#     なし
#
# [説明]
#     この関数は new() より呼び出すもので、直接呼び出し禁止です。
#     オブジェクトの属性を初期化します。
#==================================================
sub _init {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::_init';			# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);			# ????
	my $self = shift;

	$self->{'Err'} = 0;						# エラー値

	$self->{'Order'} = { };					# 実行順位
	#		->{ id } = num

	$self->{'Func'} = [ ];					# ルーチンテーブル
	#		->[ num ] = sub { ... }
}

#==================================================
# ●エラー値をセットまたはエラー値を返す
# 
# [書式]
#     err = err();					【 エラー値参照 】
#     err( err );					【 エラー値セット 】
# 
# [引数]
#     err : エラー値				【 セット時 】
# 
# [返り値]
#     err = エラー値				【 参照時 】
# 
# [説明]
#     ルーチン内のエラーをセットまたは参照します。
#==================================================
sub err {
	# printf "\n\n=== { %s } ", __PACKAGE__.'::err';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	return $self->{'Err'} if ( $#_ < 0 );			# 引数がなければエラーを返す
	$self->{'Err'} = shift;							# 引数があればエラーをセット
}

#==================================================
# ●ルーチンをセット
# 
# [書式]
#     r = set( id, func, insid )
# 
# [引数]
#        id : ルーチンID(大文字小文字は区別しない)
#      func : ルーチン
#     insid : 挿入先ルーチンID(大文字小文字は区別しない)
# 
# [返り値]
#     r :
#         (成功) : 反映したルーチンIDの実行順位
#         (失敗) : undef
# 
# [エラーコード]
#     1 : ルーチンIDが未定義またはヌル
#     2 : 挿入先ルーチンIDがヌル
#     3 : ルーチンが未定義またはコードリファレンスでない
# 
# [説明]
#     ルーチンをセットします。
#     登録するルーチンは通常のサブルーチン呼び出しと違い、引数を与えることはできません。
#     引数を与えるようにするには、オブジェクト変数(いわゆる $self のような役割を持つ変数)
#     などを介して行うようにします。
# 
#     これは例えば、
# 
#       sub routine {
#         my $self = shift;		# オブジェクト変数
#         :
#       }
# 
#     のようなルーチンを
# 
#       set( 'routine01', sub { $self->routine() } );
# 
#     のように、矢印記法で定義してセットすることで、オブジェクト変数がサブルーチンに渡るので、
#     ハンドラ呼び出しの直前でオブジェクト変数に引数となるデータをセットし、その後ハンドラを呼び出すことで
#     オブジェクト変数に含まれる引数となるデータにアクセスすることができます。
# 
#     ハンドラが呼び出されたとき、ルーチンを追加した順に実行します
# 
#     ルーチンの追加(後方追加)、挿入、更新を行うことができます。
#      set() では実行順位の変更を行うことができますが、 order() の方が簡単です。
# 
#     ルーチンの追加などの操作はルーチンIDを基準に行います。
# 
#     【 追加 】
# 
#     指定したルーチンIDがハンドラに登録されていない場合、ルーチンテーブルの後方から追加します。
# 
#      set( 'A', sub { ... } )  ;  [ A ]
#      set( 'B', sub { ... } )  ;  [ A, B ]
# 
#     【 更新 】
# 
#     指定したルーチンIDがハンドラに登録されている場合、登録されているルーチンを
#     指定したルーチンに更新します。
# 
#      set( 'A', sub { 1 } )  ;  すでに登録されているコード
#      set( 'A', sub { 2 } )  ;  新なコードに更新
# 
#     【 挿入 】

#     挿入先のルーチンIDを指定した場合、挿入先のルーチンIDが登録されているときは、
#     その直前に登録するルーチンIDを挿入してルーチンを追加します。
# 
#      set( 'A', sub { ... } )       ;  [ A ]
#      set( 'B', sub { ... } )       ;  [ A, B ]
#      set( 'C', sub { ... }, 'B' )  ;  [ A, B ] -> [ A, C, B ]
# 
#     ただし、指定する挿入先のルーチンIDが登録されていないときは、
#     挿入は行わずルーチンを追加(後方追加)します。
# 
#      set( 'D', sub { ... }, 'Z' )  ;  [ A, B ] -> [ A, B, D ]
#==================================================
sub set {
	# printf "\n\n=== { %s }", __PACKAGE__.'::set';			# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	# print "\n[\@_]\n".Dumper(@_);													# ????
	my $id = shift;				# ルーチンID
	$self->err(0);				# エラーリセット

	if ( ( defined($id) == 0 ) || ( $id eq '' ) ) {							# ルーチンIDが未定義またはヌル(1);
		$self->err(1);
		return;
	}
	$id = uc($id);
	my $func = shift;		# ルーチン
	if ( ( defined($func) == 0 ) || ( ref($func) ne 'CODE' ) ) {			# ルーチンが未定義またはコードリファレンスでない(3);
		$self->err(3);
		return;
	}

	# ●ルーチンIDの実行順位を取得
	# printf "\n** id = [ %s ]", $id;													# ????
	my $idnum = $self->{'Order'}->{ $id };
	# if ( ( defined($idnum) == 1 ) && ( $idnum >= 0 ) ) {							# 取得できた→更新
	# 	printf " -- idnum = ( %d )", $idnum;											# ????
	# }
	# else {																			# 取得できなかった→追加
	# 	print " -- (no_idnum)";														# ????
	# }

	# ●挿入先ルーチンIDと実行順位を取得
	my $insid = shift;
	my $insnum;
	if ( defined($insid) ) {														# 挿入先ルーチンIDの実行順位を取得
		if ( $insid eq '' ) {															# 挿入先ルーチンIDがヌル(2);
			$self->err(2);
			return;
		}
		$insid = uc($insid);
		# printf "\n** insid = [ %s ] ", $insid;											# ????
		$insnum = $self->{'Order'}->{ $insid };
		if ( ( defined($insnum) == 1 ) && ( $insnum >= 0 ) ) {							# 取得できた→（挿入・再配置）
			# printf "( %d )", $insnum;														# ????
		}
		else {																			# 取得できなかった→（追加）
			# print  " -- (no_insid)";														# ????
			$insid = undef;																	# 挿入先ルーチンIDをクリア
		}
	}

	##     id = ルーチンID
	##  idnum = ルーチンIDのレコード番号
	##  insid = 挿入先のルーチンID
	## insnum = 挿入先のルーチンIDのレコード番号

	my $r;																			# 返り値

	# ■挿入先ルーチンIDがない
	unless ( defined($insnum) ) {
		# ●登録済みのルーチンID→更新
		if ( defined($idnum) ) {
			# print "\n** ( update )";														# ????
 			$self->{'Func'}->[ $idnum ] = $func;											# ルーチンを更新する
			$r = $idnum;																	# 更新したレコード値を返り値に設定
		}
		# ●登録のないルーチンID→追加
		else {
			# print "\n** ( add )";															# ????
			push( @{ $self->{'Func'} }, $func );											# ルーチンテーブルに追加
			$self->{'Order'}->{ $id } = $#{ $self->{'Func'} };								# タスクテーブルに保存
			$r = $self->{'Order'}->{ $id };													# 追加したレコード値を返り値に設定
		}
	}

	# ■挿入先ルーチンIDがある
	else {
		# ●登録済みのルーチンID
		if ( defined($idnum) ) {
			# ●追加するルーチンIDと挿入先ルーチンIDが相違→実行順位変更
			unless ( $id eq $insid ) {
				# print "\n** ( reorder )";														# ????
				my $r_remove = $self->remove( $id );											# 登録済みルーチンを削除して
				# printf "\nremove.r = (%d)", $r_remove;										# ????
				# print "\n[self]\n".Dumper($self);											# ????
				my $r_insert = $self->_insert( $id, $insnum, $func );							# 挿入先ルーチンIDにルーチンを挿入
				# printf "\ninsert.r = (%d)", $r_insert;										# ????
				$r = $insnum;																	# 挿入したレコード値を返り値に設定
			}
			# ●追加するルーチンIDと挿入先ルーチンIDが同一→更新
			else {
				# print "\n** ( same_id -> update )";												# ????
	 			$self->{'Func'}->[ $idnum ] = $func;											# ルーチンテーブルを更新する
				$r = $idnum;																	# 更新したレコード値を返り値に設定
			}
		}
		# ●登録のないルーチンID
		else {

			# ●登録済みの挿入先ルーチンID→挿入
			if ( defined($insnum) ) {
				# print "\n** ( insert )";														# ????
				my $r_insert = $self->_insert( $id, $insnum, $func );								# 挿入先ルーチンIDにルーチンを挿入
				# printf "\ninsert.r = (%d)", $r_insert;											# ????
				$r = $insnum;																	# 挿入したレコード値を返り値に設定
			}
			# ●登録のない挿入先ルーチンID→追加
			else {
				# print "\n** ( -> add )";														# ????
				push( @{ $self->{'Func'} }, $func );											# ハンドラテーブルに追加
				$self->{'Order'}->{ $id } = $#{ $self->{'Func'} };								# タスクテーブルに保存
				$r = $self->{'Order'}->{ $id };													# 追加したレコード値を返り値に設定
			}
		}
	}
	return ( defined($r) ) ? $r : 999999;
}

#==================================================
# ●ルーチンを挿入
# 
# [書式]
#     r = _insert( id, insnum, func )
# 
# [引数]
#         id : ルーチンID
#     insnum : 挿入先実行順位(タスクテーブルのレコード番号)
#       func : ルーチン
# 
# [返り値]
#     r =
#         (成功) : 挿入したレコード値(insnumの値)
#         (失敗) : undef
# 
# [エラーコード]
#     1 : ルーチンIDが未定義またはヌル
#     2 : ルーチンIDが既に登録されている
#     3 : 挿入先の実行順位が未定義または数値でない
#     4 : ルーチンが未定義またはコードリファレンスでない
# 
# [説明]
#     この関数は set() より呼び出すもので、直接呼び出し禁止です。
#     指定する実行順位(タスクテーブルのレコード番号)の手前にルーチンを挿入します。
#     ルーチンの挿入方法は set() を参照ください。
#==================================================
sub _insert {
	# printf "\n\n=== { %s }", __PACKAGE__.'::_insert';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	$self->err(0);			# エラーリセット

	my $id = shift;			# ルーチンID
	if ( ( defined($id) == 0 ) || ( $id eq '' ) ) {						# ルーチンIDが未定義またはヌル(1)
		$self->err(1);
		return;
	}
	$id = uc($id);
	if ( exists( $self->{'Order'}->{ $id } ) ) {							# ルーチンIDが既に登録されている(2)
		$self->err(2);
		return;
	}
	my $insnum = shift;		# 挿入先の実行順位
	if ( ( defined($insnum) == 0 ) || ( $insnum !~ /^\d+$/ ) ) {		# 挿入先ルーチンIDが未定義または数値でない(-3)
		$self->err(3);
		return;
	}
	my $func = shift;		# ルーチン
	if ( ( defined($func) == 0 ) || ( ref($func) ne 'CODE' ) ) {		# ルーチンが未定義またはコードリファレンスでない(-4)
		$self->err(4);
		return;
	}

	# printf "\n ** id=[ %s ] ", $id;										# ????
	# printf "\n ** insnum=( %d ) ", $insnum;								# ????

	# ●採番
	foreach my $id ( keys( %{ $self->{'Order'} } ) ) {
		my $num = $self->{'Order'}->{ $id };
		# printf "\n== id=[ %s ]( %d ) ", $id, $num;									# ????
		if ( $insnum <= $num ) {														# ポインタの実行順位以降は
			$self->{'Order'}->{$id} = $num + 1;												# ＋１
			# printf ", renumber=( %d ) ", $self->{'Order'}->{$id};							# ????
		}
		# else {																		# ????
			# printf ", renumber=( %d ) ", $num;										# ????
		# }																				# ????
	}

	# ●挿入
	$self->{'Order'}->{ $id } = $insnum;									# タスクテーブルに挿入
	splice( @{ $self->{'Func'} }, $insnum, 0, $func );						# ハンドラテーブルに挿入
	return $insnum;															# 挿入したレコード値を返り値に設定
}

#==================================================
# ●ルーチンを削除
# 
# [書式]
#     r = del( id )
#
# [引数]
#     id : ルーチンID
#
# [返り値]
#     r =
#         (成功) : 削除したレコード値
#         (失敗) : undef
# 
# [エラーコード]
#     1 : ルーチンIDが未定義またはヌルもしくは登録されていない
# 
# [説明]
#     ハンドラからルーチンを削除します。
#==================================================
sub del {
	# printf "\n\n=== { %s }", __PACKAGE__.'::del';			# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	$self->err(0);			# エラーリセット

	my $id = shift;			# ルーチンID
	if ( ( defined($id) == 0 ) || ( $id eq '' ) ) {				# ルーチンIDが未定義またはヌル(1)
		$self->err(1);
		return;
	}
	$id = uc($id);
	my $idnum = $self->{'Order'}->{ $id };						# 実行順位を取得
	unless ( defined($idnum) ) {								# ルーチンIDが登録されていない(1)
		$self->err(1);
		return;
	}
	# printf "\n** id = [ %s ]( %d )", $id, $idnum;				# ????

	# ●削除
	delete( $self->{'Order'}->{ $id } );										# タスクテーブルから削除
	splice( @{ $self->{'Func'} }, $idnum, 1 );	 							# ルーチンテーブルから削除

	# ●採番
	foreach my $id ( keys( %{ $self->{'Order'} } ) ) {
		my $num = $self->{'Order'}->{ $id };
		# printf "\n== id=[ %s ]( %d ) ", $id, $num;									# ????
		if ( $idnum < $num ) {														# ポインタの実行順位以降は
			$self->{'Order'}->{ $id } = $num - 1;										# －１
			# printf ", renumber = ( %d ) ", $self->{'Order'}->{ $id };					# ????
		}
		# else {																	# ????
			# printf ", renumber = ( %d ) ", $num;									# ????
		# }																			# ????
	}
	return $idnum;																	# 削除したレコード値を返り値に設定
}

#==================================================
# ●ハンドラの実行順位の取得または設定
# 
# [書式]
#     ids = order()               【 取得 】
#     cnt = order( id, ... )      【 変更 】
# 
# [引数]
#     id : ルーチンID             【 順序変更時 】
# 
# [返り値]
#     ids = ルーチンIDリスト       【 順序取得時 】
#             [ id, ... ] (データ型: 配列リファレンス)
#     cnt = ルーチンの登録総数      【 順序変更時 】
# 
# [説明]
#     ルーチンの実行順位の取得および変更を行います。
# 
#     【 取得 】
# 
#     引数を与えず
# 
#     $ids = order();
#
#     と呼び出すと、登録順のルーチンIDが返ります。
# 
#     【 変更 】
# 
#     実行する順にルーチンIDを列挙指定することで変更します。
# 
#     例えば、実行順が
# 
#     [ Func1, Func2, Func3 ]
# 
#     を逆順に変更する場合は、
# 
#     order( Func3, Func2, Func1 );
# 
#     とします。

#     登録されていないハンドラIDを指定した場合は無効です。
# 
#     ハンドラIDを複数回指定した場合は最初の変更にのみ有効、２回目以降は無効となります。
#     例えば、
# 
#     order( 'Func3', 'Func1', 'Func2', 'Func3' );
# 
#     のように Func3 を２つ指定している場合、最初の指定は有効ですが、最後の指定は無効になりますので、
# 
#     [ Func3 ]						← 'Func3'は有効
#     [ Func3, Func1 ]				← 'Func1'
#     [ Func3, Func1, Func2 ]		← 'Func2'
#     [ Func3, Func1, Func2 ]		← 'Func3'は無効
# 
#     のようになります。
# 
# [備考]
#     ●順序変更後、無効になったルーチンを削除するようにする。
#==================================================
sub order {
	# printf "\n\n=== { %s }", __PACKAGE__.'::order';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;

	# ▼引数なし(順序出力)
	if ( $#_ < 0 ) {
		return [ sort { $self->{'Order'}->{$a} <=> $self->{'Order'}->{$b} } keys %{ $self->{'Order'} } ];
	}

	# ▼引数あり(順序変更)
	else {									
		# print "\n[args]\n".Dumper(@_);			# ????
		my $func_ = [];							# 更新後のハンドラテーブル
		my $task_ = { };							# 更新後のタスクリスト
		map {
			my $num = $self->{'Order'}->{ uc($_) };
			my $num_ = $task_->{ uc($_) };
			if ( ( defined( $num ) == 1 ) && ( defined( $num_ ) == 0 ) ) {		# ハンドラIDが登録済みかつ未変更
				push( @{ $func_ }, $self->{'Func'}->[ $num ] );					# ルーチンテーブルにコピー
				$task_->{ uc($_) } = $#{ $func_ };								# 順位をタスクリストに保存
			}
		} @_;
		# print "\n[handler_]\n".Dumper( $handler_ );		# ????
		# print "\n[task_]\n".Dumper( $task_ );				# ????

		$self->{'Func'} = $func_;					# ルーチンテーブルの更新
		$self->{'Order'} = $task_;					# タスクリストの更新
		return $#{ $self->{'Func'} } + 1;			# 残ったルーチン総数
	}
}

#==================================================
# ●登録したルーチンを呼び出す
# 
# [書式]
#     call( id, ... )
#
# [引数]
#     id : 呼び出すルーチンID (省略時: 登録順に全て呼び出す )
#
# [返り値]
#     なし
# 
# [説明]
#     ルーチンIDを列挙指定すると指定した順にルーチンを呼び出します。
#     登録されていないルーチンIDが指定されている場合は無視されます。
#     また、登録されているルーチンIDが列挙指定されていない場合、そのルーチンは呼び出しを行いません。
# 
#     ハンドラに登録するルーチンは通常のサブルーチン呼び出しと違い、
#     ルーチン対して引数を与えることはできませんので、引数を与えるようにするには
#      call() の前に引数を保存し、呼び出したハンドラのルーチン側で引数を受け取るような
#     処理を定義してください。
#==================================================
sub call {
	# printf "\n\n=== { %s }", __PACKAGE__.'::call';		# ????
	# printf "\n=== ( %s )", Debug::_caller(caller);		# ????
	my $self = shift;
	$self->err(0);								# エラーリセット
	# print "\n[self]\n".Dumper( $self );			# ????

	if ( $#{ $self->{'Func'} } < 0 ) {
		# print "\n** (no_func)";									# ????
		$self->err(1);											# ハンドラルーチンが登録されていない(1)
		return;
	};

	my @task;

	if ( $#_ >= 0 ) {											# 引数あり
		@task = map { $self->{'Order'}->{ uc($_) } } @_;
		# print "\n[id]\n".Dumper(@_);							# ???? 引数の一覧
	}
	else {														# 引数なし
		@task = ( 0..$#{ $self->{'Func'} } );
		# print "\n** ( no_ids )";				# ???? 引数なし
	}

	# my @task = ( $#_ >= 0 )
	# 			? map { $self->{'Order'}->{ uc($_) } } @_			# 引数あり
	# 			: ( 0..$#{ $self->{'Func'} } );						# 引数なし

	# print "\n[task]\n".Dumper(@task)."\n";								# ????

	# ルーチンを呼び出す
	map {
		my $num = $_;
		if ( defined($num) ) {
			my ( $id ) = grep { $self->{'Order'}->{$_} == $num } keys( %{ $self->{'Order'} } );
			# printf "\n************* [ %d : %s ] *************", $num, $id;			# ????

			# eval {
				&{ $self->{'Func'}->[ $num ] };							# ルーチンを呼び出す
			# };

			# if ($@) {													# ルーチン内でエラーが発生している(1)
			# 	$self->err( 1 );
			# 	# print $@;												# ????
			# }
		}
		# else {
		# 	# printf "\n************* ( no_id  : %s ) *************", $num;			# ????
		# }
	} @task;
	return;
}

printf("\n(%s) [ %s ] ", $VERSION->[-1], __PACKAGE__ );

#===========================================================
=pod
=encoding utf8
=head1 スクリプト名
Handler.pm - BBS.pm サブモジュール【 ハンドラ制御 】
=head1 著者
naoit0 <https://www.naoit0.com/projects/bbs_pm/>
=cut
1;