package BBS;
;#===========================================================
# BBS.pm コアシステム
#
our $VERSION = [
	"0001.20250322.2045",		# 正式公開版
];
#===========================================================
# [ 説明 ]
#     アプリケーションサーバの作成は本モジュールをロードして、各ハンドラの関数をオーバーライドすることで機能します。
#     詳細はドキュメントをご覧ください。
#===========================================================
use utf8;
use strict;
use warnings;
use IO::Select;
use IO::Socket;
use Scalar::Util;
use Encode qw( encode decode );
use Data::Dumper;
use Debug;

use BBS::Node;							# ノード
use BBS::Handler;						# ハンドラ
use BBS::Timer;							# タイマー

#==================================================
# ●コンストラクタ( new )
# 
# [書式]
#     obj = new()
# 
# [引数]
#     なし
# 
# [返り値]
#     obj =
#         (成功) : オブジェクト
#         (失敗) : undef
# 
# [エラーコード]
#     なし
# 
# [説明]
#     オブジェクトを作成します。
# 
# [メモ]
# 
#==================================================
sub new {
	my $this = shift;
	my $class = ref($this) || $this;
	my $self = { @_ };
	bless( $self, $class );
 	$self->_init();

	return $self;
}

#==================================================
# ●属性の初期化( _init )
# 
# [書式]
#     _init()
# 
# [引数]
#     なし
# 
# [返り値]
#     なし
# 
# [エラーコード]
#     なし
# 
# [説明]
#     オブジェクトの属性を初期化します。
# 
# [メモ]
#     この関数は new() より呼び出すもので、直接呼び出し禁止です。
#==================================================
sub _init {
	my $self = shift;

	# エラーコード
	$self->{'err'}				= 0;

	# セレクタ
	$self->{'select'}			= undef;

	# イベントを発生したノード番号
	$self->{'from'}				= undef;

	# サーバ停止フラグ
	$self->{'sigterm'}			= 0;

	# ロックファイル
	$self->{'lockfile'}			= ".lock";

	# 受信データのサイズ
	$self->{'rcvsize'}			= 4096;

	# 受信データ
	$self->{'recv'}				= '';

	#*****************************
	# ノードテーブル( Node )
	# nodelist(), node(), from_node() を追加。
	#*****************************

	$self->{'__Node'}			= new BBS::Node();
	# ※以下のパラメータ名はシステムで使用します。
	# 同じパラメータ名は使用しないでください。
	# 				->node(n)
	#					->{'__Socket'}			= $sock;					# ソケットオブジェクト
	#					->{'peerhost'}			= $socket->peerhost();		# ノードのアドレス
	#					->{'peerport'}			= $socket->peerport();		# ノードのポート
	#					->{'Disconnected'}		= n;						# 切断イベントフラグ
}

#==================================================
# ●エラー値をセットまたはエラー値を返す( err )
# 
# [書式]
#     err = err();					【 エラー値参照 】
#     err( err );					【 エラー値セット 】
# 
# [引数]
#     err : エラー値				【 セット時 】
# 
# [返り値]
#     err = エラー値				【 参照時 】
# 
# [説明]
#     ルーチン内のエラーをセットまたは参照します。
# 
# [メモ]
# 
#==================================================
sub err {
	my $self = shift;
	return $self->{'err'} if ( $#_ < 0 );			# 引数がなければエラーを返す
	$self->{'err'} = shift;							# 引数があればエラーをセット
}

#==================================================
# ●接続ノードリストを得る( nodelist )
# 
# [書式]
#     list = nodelist()
#
# [引数]
#     なし
#
# [返り値]
#     list = ノード番号(データ型: 配列)
#            ( node1, node2, ... )
#            (要素: ノード番号, 順序: 接続順)
#
# [説明]
#     接続しているノードリストを取得します。
#==================================================
sub nodelist {
	my $self = shift;
	my @r = $self->{'__Node'}->nodelist();						# 接続順リストから取得
	# printf "\n** nodelist = [ %s ] ", join(',', @r);			# ???? 接続リスト
	return @r;
}

#==================================================
# ●ノードテーブルのデータ領域を返す( node )
# 
# [書式]
#     r = node( num )
#
# [引数]
#     num : ノード番号
#
# [返り値]
#     r =
#         (成功) : データ領域
#         (失敗) : undef
#
# [エラーコード]
#     1 : 引数が無効
#     2 : ノードが開かれていない
# 
# [説明]
#     ノードテーブルのデータ領域を返します。
#
# [メモ]
#     ノードテーブルはそれぞれのノードが保持するデータを保持する領域で、
#     新たなノードが接続するとノードのためのデータ領域がノードテーブルに用意されます。
#     接続中のノードが切断を試みるとノードが保持するデータ領域がノードテーブルから削除されます。
#
#     ノードテーブルはハッシュ変数ですので、返り値に対して操作することで、データを追加、削除することができます。
#
#     ノードテーブルはシステムが使用するデータも保存されるため、
#     誤ってシステムデータを削除したり書き換えたりしないよう、十分注意する必要があります。
#     システムが使用するデータは関数 _init() のノードテーブルの項目に記されています。
#==================================================
sub node {
	my $self = shift;
	my $num = shift;			# ノード番号
	$self->err( 0 );			# エラーリセット

	if ( ( defined( $num ) == 0 ) || ( $num !~ /^\d+$/ ) ) {
		$self->err( 1 );		# 引数が無効(1)
		return;
	}

	my $node = $self->{'__Node'}->node( $num );
	unless ( defined( $node ) ) {
		$self->err( 2 );		# ノードが開かれていない(2)
		return;
	};

	return $node;				# データ領域を返す

}

#==================================================
# ●ノードテーブルのデータ領域を返す( from_node )
# 
# [書式]
#     r = from_node()
#
# [引数]
#     なし
#
# [返り値]
#     r =
#         (成功) : データ領域
#         (失敗) : undef
#
# [エラーコード]
#     2 : ノードが開かれていない
# 
# [説明]
#     ノードテーブルのデータ領域を返します。
#
# [メモ]
#     返すデータ領域はイベントが発動したノードが対象です。
#==================================================
sub from_node {
	my $self = shift;
	my $num = $self->from();
	return $self->node( $num );				# データ領域を返す
}

#==================================================
# ●イベントが発生したときに応対するノード番号を返す( from )
# 
# [書式]
#     num = from()
#
# [引数]
#    なし
#
# [返り値]
#    num = ノード番号
# 
# 
# [エラーコード]
#    なし
# 
# [説明]
#     イベント発生元のノード番号を返します。
#
# [メモ]
#    onConnect, onDisconnect, onRecv, onServeなど、
#    何らかのイベントが発生したときに、ノード番号が変数にセットされ、
#    セットされたノード番号は本関数で取得することができます。
# 
#    from 付き関数( from_send(), from_node() )は、
#    それぞれの処理を行う際、対象ノードが自動的にセットされるため、引数を省略できます。
#==================================================
sub from {
	my $self = shift;
	# printf "\n** from = (%d)", $self->{'from'};				# ????
	return $self->{'from'};
}

#==================================================
# ●サーバの開始( start )
# 
# [書式]
#     start( port )
# 
# [引数]
#    port : TCPポート番号
# 
# [返り値]
#    なし
# 
# [エラーコード]
#    なし
# 
# [説明]
#    サーバを開始し、指定したポート番号でノードからの着信を待ちます。
# 
# [メモ]
#    ポートチェックは行っていないため、
#    既に開いているポートを指定して開始することはできますが、正常に動作しない可能性があります。
#==================================================
sub start {
	my $self = shift;
	my $port = shift;
	if ( (defined($port) != 1 ) || ($port !~ /^\d+$/) ) {			# ポート番号が無効
		print "(error): ポート番号を指定してください.\n";
		exit( 0 );
	};

	# (1) ロックファイル作成
	open( my $fh, ">$self->{'lockfile'}" );
	close( $fh );

	# (2) 初期処理ハンドラ呼出
	$self->initalize();

	# (3) サーバ処理開始
	my $listen = new IO::Socket::INET( Listen => 1, LocalPort => $port );			# ソケットリスナー(3)
	$self->{'select'} = new IO::Select($listen);									# セレクタ

	$| = 1;
	while ( $self->{'sigterm'} == 0 ) {							# 終了シグナルが発生したらループを抜ける
		my @ready = $self->{'select'}->can_read( 0.0001 );			# 通信イベントを監視
		if ( $#ready >= 0 ) {										# ●通信発生
			foreach my $socket ( @ready ) {								# 行列待ちを捌く
				# printf "\n** from=(%d)\n", $self->from();				# ???? ノード番号(ファイル番号)

				if ( $socket == $listen ) {								# 【 新たなクライアントが接続した( onConnect ) 】
					$socket = $listen->accept();							# リスナーをacceptして新しいソケットを作り
					$self->{'from'} = $socket->fileno;						# ソケットのファイル番号をfrom()にセット
					$self->{'select'}->add( $socket );						# そのソケットをセレクタに追加
					$self->_onconnect( $socket );							# 接続イベント( _oncconnect )を呼び出す
				}
				else {													# 【 接続するクライアントから通信が発生した 】
					my $data;												# 受信データ
					$self->{'from'} = $socket->fileno;						# ソケットのファイル番号をfrom()にセット
					$socket->recv( $data, $self->{'rcvsize'} );				# 受信データを取得
					if ( $data eq '' ) {									# 受信データがない( onDisconnect )
						$self->_ondisconnect( $self->from() );					# 切断イベント( _ondisconnect )を呼び出す
					}
					else {													# 受信データがある( onRecv )
						$self->{'recv'} = $data;								# 受信データを受信バッファに保存
						$self->input();											# 受信ハンドラ( input )を呼び出す
						$self->{'recv'} = undef;								# 受信バッファをクリア
					}
				}
				$self->_onserve( $self->from() );						# サーバイベント( _onserve )へ
			}
		}
		else {														# ●待機中
			$self->_onserve();												# サーバイベント( _onserve )へ
		}
	}																# 停止フラグが立たなければループに戻る

	# (4) 終了処理ハンドラ呼出
	$self->finalize();
	return 0;														# 停止フラグが立たてばサーバを停止
}

#==================================================
# ●受信データ取得( recv )
# 
# [書式]
#     data = recv()
#
# [引数]
#    なし
#
# [返り値]
#    data : 受信データ
#
# [エラーコード]
#    なし
#
# [説明]
#     受信データの取得を行います。
#     本関数は受信ハンドラ処理( input )内でのみ有効で、それ以外で呼び出した場合、
#     未定義( undef )が返ります。
#==================================================
sub recv {
	# printf "\n\n{ %s }", __PACKAGE__.'::recv';		# ????
	my $self = shift;
	return $self->{'recv'};
}

#==================================================
# ●送信処理( send )
# 
# [書式]
#     send( to, data )
#
# [引数]
#     to : 送信先ノード
#   data : 送信データ
#
# [返り値]
#    なし
#
# [エラーコード]
#    なし
#
# [説明]
#     データの送信を行います。
#==================================================
sub send {
	my $self = shift;
	# print "\n[ arg ]\n".Dumper( @_ );					# ????

	my $to = shift;											# 送信先ノード
	my $data = shift;										# 送信データ
	my $socket = $self->node( $to )->{'__Socket'};			# ソケットオブジェクト

	return if ( ( defined( $to ) == 0 ) || ( $to !~ /^\d+$/ ) )		# 送信先ノードが未定義または数値でない
		or ( ( defined( $data ) == 0 ) || ( $data eq '' ) )			# 送信データが未定義またはヌル
 		or ( ref($socket) ne 'IO::Socket::INET' );					# ソケットオブジェクトでない

	$socket->send( $data );											# データ送出
}

#==================================================
sub from_send {
	my $self = shift;

	return $self->send( $self->from(), @_ );								# イベント発生元ノードに対してデータ送信
}

#==================================================
# ●切断処理( disconnect )
# 
# [書式]
#     disconnect( node )
# 
# [引数]
#     node : 対象ノード番号
# 
# [返り値]
#    なし
# 
# [エラーコード]
#     1 : ノード番号が無効
#     2 : ノードが開かれていない
#     3 : すでに切断イベントを通過している
# 
# [説明]
#     ノードとの切断処理を行います。
# 
# [メモ]
#     この関数はログアウト処理など、ノード側の要求または状態によって切断処理を行うためのもので、
#     処理は単純に切断イベント処理( _ondisconnect() )を呼び出しています。
#     ノードとの接続を切断するときに本関数を呼び出します。
#==================================================
sub disconnect {
	my $self = shift;
	my $node = shift;

	return $self->_ondisconnect( $node );
}

#==================================================
# ●サーバ処理( _onserve )
# 
# [書式]
#    _onserve( nodeno )
#
# 
# [引数]
#    nodeno : 処理を開始するノード番号(省略可)
# 
# [返り値]
#    なし
# 
# [エラーコード]
#    なし
# 
# [説明]
#    アプリケーションサーバにおける出力処理( Output )と応答処理( Appwork )を行います。
# 
# [メモ]
#     この関数は start() より呼び出すもので、直接呼び出し禁止です。
# 
#     ノードごとに次のハンドラが呼び出されます。
# 
#       (1) ノードにデータを送出する出力処理(Output)
#       (2) アプリケーション処理(AppWork)
# 
#     処理するノードの優先順位は先着接続順または指定するノードを基準として接続順で行われます。
#     先着接続順は、接続しているノード接続順が、
#       [ A, B, D, C ]
#     なら、ノードの接続順のとおり、
#       A → B → D → C
#     の順番で処理が行われます。
#     指定するノードを基準とした接続順はノード接続順は上記の通りで、ノード D を基準とした場合、
#       D → C → A → B
#     の順番で処理が行われます。
#     全ノードを平等にサーバ処理を行うため、イベントが発生した場合、対象ノードを基準とし、
#     イベントが発生しなかった場合は先着接続順にサーバ処理を行うようにしています。
#==================================================
sub _onserve {
	# printf "\n\n{ %s }", __PACKAGE__.'::_onserve';		# ????
	my $self = shift;

	my $nodeno = shift;				# 呼び出し開始ノード
	my @served;						# 実行済みリスト( $served[ nodeno ] = flag )
	my $exit = 0;					# 終了フラグ
	my $running = 0;				# 実行中ノード番号(デフォルト:0)

	$self->{'sigterm'} = 1 unless ( -e "$self->{'lockfile'}" );							# ロックファイルが消失したら終了シグナルを発生、プログラムを停止

   	my @nodelist = $self->nodelist();												# 接続順リスト取得
	return if ( $#nodelist < 0 );															# ノードが一つも接続されていない(undef)

	if ( defined($nodeno) ) {															# 呼び出し開始ノードが指定されていたら、指定ノードの順番（順位）を取得
		my ($order) = grep { $nodelist[ $_ ] eq $nodeno } 0..$#nodelist;							# [ 4, 5, 7, 6 ] ; 4 = [0] ; 5 = [1] ; 7 = [3] ; 6 = [4]
		$running = $order if ( defined($order) );											# 順位が得られたら、実行中ノード番号を更新
	}

	# ●ノードに対してサーバ処理を行う
	until ( $exit ) {																	# 終了フラグが立ったら終了
		# printf "\n\n** node = ( %d ), order = [ %d ], exit = ( %d )"
		# 				, $nodelist[$running], $running, $exit;								# ???? 実行中のノード番号, ノードの実行順位, 終了フラグ

		$self->{'from'} = $nodelist[ $running ];											# 実行中のノード番号をfrom()にセット
		$self->output();																	# 出力処理(Output)
		$self->appwork();																	# アプリケーション処理(AppWork)
		# $self->output();																	# 出力処理(Output)
		$served[ $running ] = 1;															# 実行済みリストにノード番号を追加

		# 巡回終了チェック
		$running++;																			# ノードの実行順位＋１
		$running = ( $running > $#nodelist ) ? 0 : $running;								# ノードの実行順位がノード総数を超えたら０をセット
		$exit = 1 if ( defined( $served[ $running ] ) );									# 実行するノードが実行済みリストに追加されているなら終了
		# print "\n";																		# ????
		# print " ";
	}
	# printf "\n** ( onserve_exit )";															# ???? サーバ処理終了
}

#==================================================
# ●接続イベント処理( _onconnect )
# 
# [書式]
#     _onconnect( socket )
# 
# [引数]
#     sokect : ソケットオブジェクト
# 
# [返り値]
#     なし
# 
# [エラーコード]
#     1 : 引数が未定義またはコードリファレンスでない
# 
# [説明]
#     アプリケーションサーバにおけるノードとの接続処理を行います。
# 
# [メモ]
#     この関数は start() より呼び出すもので、直接呼び出し禁止です。
#     本関数は新たなノードがサーバに接続すると呼び出されます。
#     この関数ではノードの追加と接続したソケットのアドレスおよび
#     ポート番号をノードテーブルに保存した後、接続ハンドラ(logon)を呼び出します。
#==================================================
sub _onconnect {
	my $self = shift;
	my $socket = shift;					# ソケットオブジェクト
	$self->err( 0 );					# エラーリセット

	if ( ( defined($socket) == 0 ) || ( ref($socket) ne 'IO::Socket::INET' ) ) {
		$self->err( 1 );											# 引数が未定義またはソケットオブジェクトでない(1)
		return;
	}

	my $me = $self->from();
	# printf "\n** me = ( %d ) ", $me;								# ???? 対象のノード番号

	$self->{'__Node'}->add( $me );									# 接続ノードをノードテーブルに追加
	$self->node( $me )->{'__Socket'} = $socket;						# ソケットオブジェクトを保存
	$self->node( $me )->{'peerhost'} = $socket->peerhost();			# ノードのアドレスを保存
	$self->node( $me )->{'peerport'} = $socket->peerport();			# ノードのポートを保存
	$self->logon( $me );											# Logon ハンドラを呼び出す
}

#==================================================
# ●切断イベント処理( _ondisconnect )
# 
# [書式]
#     _ondisconnect( node )
# 
# [引数]
#     node : 対象ノード番号
# 
# [返り値]
#    なし
# 
# [エラーコード]
#     1 : ノード番号が無効
#     2 : ノードが開かれていない
#     3 : すでに切断イベントを通過している
# 
# [説明]
#     アプリケーションサーバにおけるノードとの切断処理を行います。
# 
# [メモ]
#     この関数は start() より呼び出すもので、直接呼び出し禁止です。
#     本関数は接続中ノードが切断すると呼び出されます。
#     本関数では切断ハンドラ(logoff)を呼び出した後、ノードの廃棄を行います。
#==================================================
sub _ondisconnect {
	my $self = shift;
	my $node = shift;						# ノード番号
	printf "\n** node = ( %d )", $node;		# ???? 対象のノード番号
	$self->err( 0 );						# エラーリセット

	if ( ( defined( $node ) == 0 ) || ( $node !~ /^\d+$/ ) ) {
		$self->err( 1 );					# ノード番号が無効(1)
		return;
	}
	unless ( defined( $self->node( $node ) ) ) {
		$self->err( 2 );					# ノードが開かれていない(2)
		return;
	};

	# ●切断イベントの二重呼び出しチェック
	my $me = $self->node( $node );
	my $disconnected = $me->{'Disconnected'};		# 切断イベントフラグを参照
	if ( defined( $disconnected ) ) {
		$self->err( 3 );	 							# すでに切断イベントを通過している(3)
		return;
	};
	$me->{'Disconnected'} = 1;						# 切断イベントフラグを立てる

	# ●Logoff ハンドラ呼び出し
	$self->logoff( $node );							# logoff ハンドラを呼び出す

	# ノードの整理
	my $socket = $self->node( $node )->{'__Socket'};
	$socket->close;									# ソケットを閉じる
	$self->{'select'}->remove( $socket );			# セレクタからソケットを削除
	my $n = $self->{'__Node'};
	$n->del( $node );								# オーダリストから削除
	$n->remove( $node );							# ノードを廃棄
}

#==================================================
# ●初期処理ハンドラ処理( initalize )
# 
# [書式]
#     initalize()
# 
# [引数]
#     なし
# 
# [返り値]
#    なし
# 
# [エラーコード]
#    なし
# 
# [説明]
#     アプリケーションサーバの初期処理を定義します。
# 
# [メモ]
#     処理はこの関数をオーバーライドします。
#==================================================
sub initalize {
	printf "\n\n-- { %s }", __PACKAGE__.'::initalize';			# ????
	my $self = shift;

	# 【 サーバの初期処理をここに定義 】
}

#==================================================
# ●終了処理ハンドラ処理( finalize )
# 
# [書式]
#     finalize()
# 
# [引数]
#     なし
# 
# [返り値]
#    なし
# 
# [エラーコード]
#    なし
# 
# [説明]
#     アプリケーションサーバの終了処理を定義します。
# 
# [メモ]
#     処理はこの関数をオーバーライドします。
#==================================================
sub finalize {
	printf "\n\n-- { %s }", __PACKAGE__.'::finalize';			# ????
	my $self = shift;

	# 【 サーバの終了処理をここに定義 】
}

#==================================================
# ●接続ハンドラ処理( logon )
# 
# [書式]
#     logon( node )
# 
# [引数]
#     node = 接続したノード番号(省略可)
# 
# [返り値]
#    なし
# 
# [エラーコード]
#    なし
# 
# [説明]
#     アプリケーションサーバに接続するノードに対する初期処理を定義します。
# 
# [メモ]
#     ハンドラ呼び出しの際、接続するノード番号を省略した場合、
#     ハンドラ側で関数 from() を呼び出すことで、接続したノード番号を取得することができます。
#     処理はこの関数をオーバーライドします。
#==================================================
sub logon {
	my $self = shift;
	my $node = shift;
	# printf "\n** node = ( %d ) ", $node;							# ???? 対象のノード番号

	# 【 サーバとの接続処理をここに定義 】
}

#==================================================
# ●切断ハンドラ処理( logoff )
# 
# [書式]
#     logoff( node )
#
# [引数]
#     node = 切断するノード番号(省略可)
#
# [返り値]
#    なし
# 
# [エラーコード]
#    なし
# 
# [説明]
#     アプリケーションサーバから切断するノードに対する終了処理を定義します。
#
# [メモ]
#     ハンドラ呼び出しの際、切断するノード番号を省略した場合、
#     ハンドラ側で関数 from() を呼び出すことで、切断するノード番号を取得することができます。
#     処理はこの関数をオーバーライドします。
#==================================================
sub logoff {
	# printf "\n\n{ %s }", __PACKAGE__.'::logoff';		# ????
	my $self = shift;

	my $node = shift;								# 切断するノード番号(引数による取得)
	# my $node = $self->from();						# 切断するノード番号(関数による取得)
	printf "\n** node = ( %d ) ", $node;			# ????

	# 【 ユーザに対する切断処理ここに定義 】
}

#==================================================
# ●受信ハンドラ処理( input )
# 
# [書式]
#     input()
#
# [引数]
#    なし
#
# [返り値]
#    なし
#
# [エラーコード]
#    なし
#
# [説明]
#     アプリケーションサーバにおける受信処理を行います。
#
# [メモ]
#     ノードからデータを受信すると本関数が呼び出されます。
#     受信データの取得は関数 recv() を呼び出します。この関数は本ハンドラでのみ有効で、
#     それ以外のところでは未定義が返ります。
#     呼び出しの際、受信データが第１引数にセットされますので、ハンドラ処理で取得します。
#     処理はこの関数をオーバーライドします。
#==================================================
sub input {
	printf "\n\n{ %s }", __PACKAGE__.'::input';		# ????
	my $self = shift;

	my $data = $self->recv();		# 受信データ取得

	# 【 受信データの取得処理をここに定義 】

	# my $me = $self->from();
	# printf "\n** from = ( %d )", $me;								# ???? 対象のノード番号
	# printf "\n** data.length = ( %d )", length( $data );			# ???? 受信データのデータ長
	# printf "\n** data = [ %s ]", $data;							# ???? 受信データの内容
}

#==================================================
# ●出力ハンドラ処理( output )
# 
# [書式]
#     output()
#
# [引数]
#    なし
#
# [返り値]
#    なし
#
# [エラーコード]
#    なし
# 
# [説明]
#     アプリケーションサーバにおけるノードに対する出力処理を行います。
#
# [メモ]
#     処理はこの関数をオーバーライドします。
#==================================================
sub output {
	my $self = shift;

	my $me = $self->from();
	# printf "\n** from = ( %d )", $me;						# ???? 実行対象のノード番号
	my $data;												# 送出データ

	# 【 アプリケーション処理（ユーザへの出力処理）をここに定義 】

	# print "\n[Dump]\n".Debug::dump( $data );				# ????
	$self->from_send( $data );								# データ送出
}

#==================================================
# ●アプリケーション処理ハンドラ処理( appwork )
# 
# [書式]
#     appwork()
# 
# [引数]
#     なし
# 
# [返り値]
#     なし
# 
# [エラーコード]
#     なし
# 
# [説明]
#     アプリケーションサーバに接続するノードに対する応答処理を定義します。
#
# [メモ]
#     処理はこの関数をオーバーライドします。
#==================================================
sub appwork {
	my $self = shift;

	my $me = $self->from();
	# printf "\n** from = ( %d )", $me;			# ???? 実行対象のノード番号

	# 【 アプリケーション処理（ユーザへの応答処理）をここに定義 】
}

#==================================================
# ●CTRL+Cによるプログラム停止を無効にする( termtrap )
# 
# [書式]
#    termtrap()
#
# [引数]
#    なし
#
# [返り値]
#    なし
# 
# [エラーコード]
#    なし
# 
# [説明]
#    CTRL+Cによるプログラム停止を無効にします。
# 
# [メモ]
#    一度呼び出すと機能を取り消すことができません。
#    プログラムを停止する場合は、実行するメインプログラムのパスに作成されるロックファイルを削除します。
#==================================================
sub termtrap {
	my $self = shift;
	$SIG{'ALRM'} = sub { return };
	$SIG{'INT'}  = sub { return };
	$SIG{'HUP'}  = sub { return };
	$SIG{'QUIT'} = sub { return };
	$SIG{'TERM'} = sub { return };
	return;
}

#==================================================
# ●プロジェクト名を表示
# 
# [書式]
#    about()
#
# [引数]
#    なし
#
# [返り値]
#    なし
# 
# [エラーコード]
#    なし
#==================================================
sub about {
	my $self = shift;

	print <<EOT;
BBS.pm:
https://www.naoit0.com/projects/bbs_pm
EOT
}

printf("\n(%s) [ %s ] ", $VERSION->[-1], __PACKAGE__ );

#===========================================================
=pod
=encoding utf8
=head1 スクリプト名
BBS.pm - BBS.pm コアシステム
=head1 著者
naoit0 <https://www.naoit0.com/projects/bbs_pm/>
=cut
1;